<?php

namespace App\Commands;

use FluidXml\FluidXml;
use FluidXml\FluidNamespace;
use App\Models\FraccionArancelaria;
use Genkgo\Xsl\XsltProcessor;
use Genkgo\Xsl\Cache\NullCache;

class XMLUtils {
	public static function createXML32($dataFactura) {


		$cfdiNS = new FluidNamespace ( 'cfdi', 'http://www.sat.gob.mx/cfd/3' );
		$xsiNS = new FluidNamespace ( 'xsi', 'http://www.w3.org/2001/XMLSchema-instance' );
		$xsiSLNS = new FluidNamespace ( 'schemaLocation', 'http://www.sat.gob.mx/sitio_internet/cfd/3/cfdv33.xsd' );
		$xslNs = new FluidNamespace ( 'xsl', 'http://www.w3.org/1999/XSL/Transform' );
		$tfdNs = new FluidNamespace ( 'tfd', 'http://www.sat.gob.mx/TimbreFiscalDigital' );

		$xml = new FluidXml ( null ); // A document without a root node.
		$xml->namespace ( $cfdiNS, $xslNs ); // Let's register the namespaces associated with our document.
		$xml->namespace ( $xsiNS, $xslNs );
		$xml->namespace ( $tfdNs, $xslNs );

		$root = $xml->addChild ( 'cfdi:Comprobante', true );

		$rootNode = $root [0];

		// Agrenado namespaces
		$rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$cfdiNS->id()}", $cfdiNS->uri () );
		$rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$xslNs->id()}", $xslNs->uri () );
		$rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$xsiNS->id()}", $xsiNS->uri () );
		$rootNode->setAttribute ( 'xsi:schemaLocation', "http://www.sat.gob.mx/cfd/3 http://www.sat.gob.mx/sitio_internet/cfd/3/cfdv33.xsd" );

		$fechaEmision = date_format ( new \DateTime ( $dataFactura->timeStamp ), 'Y-m-d\TH:i:s' );
		if ($dataFactura->tipo == 4) {
			$tipoDeComprobante = "egreso";
		} else {
			$tipoDeComprobante = "ingreso";
		}

		// Creando el nodo Comprobante
		$rootNode->setAttribute ( 'version', '3.2' );
		$rootNode->setAttribute ( 'serie', $dataFactura->serie );
		$rootNode->setAttribute ( 'folio', $dataFactura->folio );
		$rootNode->setAttribute ( 'fecha', $fechaEmision );
		$rootNode->setAttribute ( 'noCertificado', $dataFactura->noCertificado );
		$rootNode->setAttribute ( 'certificado', $dataFactura->certificado );
		$rootNode->setAttribute ( 'subTotal', Utils::formatNumber ( $dataFactura->mntBase, $dataFactura->precission ) );
		$rootNode->setAttribute ( 'Moneda', $dataFactura->moneda );
		$rootNode->setAttribute ( 'total', Utils::formatNumber ( $dataFactura->total, $dataFactura->precission ) );
		$rootNode->setAttribute ( 'tipoDeComprobante', $tipoDeComprobante );
		$rootNode->setAttribute ( 'formaDePago', $dataFactura->formaPago );
		$rootNode->setAttribute ( 'metodoDePago', $dataFactura->metodoDePago );
		$rootNode->setAttribute ( 'condicionesDePago', $dataFactura->condicionesDePago );
		$rootNode->setAttribute ( 'sello', $dataFactura->sello );

		if($dataFactura->cuenta != "" ){
			$rootNode->setAttribute ('NumCtaPago', $dataFactura->cuenta);
		}

		if ($dataFactura->mntDcto > 0) {

			$montoDescuento = Utils::formatNumber ( $dataFactura->mntDcto, $dataFactura->precission );
			$rootNode->setAttribute ( 'descuento', $montoDescuento );

			if ($dataFactura->motivoDescuento != '') {
				$rootNode->setAttribute ( 'motivoDescuento', $dataFactura->motivoDescuento );
			}
		}

		$rootNode->setAttribute ( 'TipoCambio', '1.00' );
		$rootNode->setAttribute ( 'LugarExpedicion', $dataFactura->lugarExpedicion );

		$impuestosRetenidos = array ();
		$impuestosTrasladados = array ();
		$totalImpuestosRetenidos = 0;
		$totalImpuestosRetenidosLocales = 0;

		$impuestosRetendosLocales = array ();
		$impuestosTrasladadosLocales = array ();
		$totalImpuestosTraslados = 0;
		$totalImpuestosTrasladosLocales = 0;

		foreach ( $dataFactura->impuestos as $impuesto ) {

			// Retenidos
			if ($impuesto->tipo == 0) {

				if ($impuesto->tipoImp == "IVA" || $impuesto->tipoImp == "ISR") {
					$impuestosRetenidos [] = $impuesto;
				} else {
					// Impuestos retenidos locales
					$impuestosRetendosLocales [] = $impuesto;
					$totalImpuestosRetenidosLocales += $impuesto->montoImp;
				}

				$totalImpuestosRetenidos += $impuesto->montoImp;
			} else {
				// Trasladados
				if ($impuesto->tipoImp == "IVA" || $impuesto->tipoImp == "IEPS") {
					$impuestosTrasladados [] = $impuesto;
				} else {
					// Impuestos trasladados locales
					$impuestosTrasladadosLocales [] = $impuesto;
					$totalImpuestosTrasladosLocales += $impuesto->montoImp;
				}

				$totalImpuestosTraslados += $impuesto->montoImp;
			}
		}

		// Agregando nodo emisor
		$domicilioEmisor = array ();
		if (! empty ( $dataFactura->calle )) {
			$domicilioEmisor ["calle"] = $dataFactura->calle;
		}

		if (! empty ( $dataFactura->codigoPostal )) {
			$domicilioEmisor ["codigoPostal"] = $dataFactura->codigoPostal;
		}

		if (! empty ( $dataFactura->colonia )) {
			$domicilioEmisor ["colonia"] = $dataFactura->colonia;
		}

		if (! empty ( $dataFactura->estado )) {
			$domicilioEmisor ["estado"] = $dataFactura->estado;
		}

		if (! empty ( $dataFactura->localidad )) {
			$domicilioEmisor ["localidad"] = $dataFactura->localidad;
		}

		if (! empty ( $dataFactura->municipio )) {
			$domicilioEmisor ["municipio"] = $dataFactura->municipio;
		}

		if (! empty ( $dataFactura->nroExterior )) {
			$domicilioEmisor ["noExterior"] = $dataFactura->nroExterior;
		}

		if (! empty ( $dataFactura->nroInterior )) {
			$domicilioEmisor ["noInterior"] = $dataFactura->nroInterior;
		}

		if (! empty ( $dataFactura->pais )) {
			$domicilioEmisor ["pais"] = $dataFactura->pais;
		}

		$root->addChild ( 'cfdi:Emisor', true, [
				'nombre' => $dataFactura->nmbEmisor,
				'rfc' => $dataFactura->rFCEmisor
		] )->addChild ( 'cfdi:DomicilioFiscal', '', $domicilioEmisor )->addChild ( 'cfdi:RegimenFiscal', '', [
				'Regimen' => $dataFactura->regimenFiscal
		] );

		// Agregando nodo emisor
		$domicilioReceptor = array ();
		if (! empty ( $dataFactura->calleR )) {
			$domicilioReceptor ["calle"] = $dataFactura->calleR;
		}

		if (! empty ( $dataFactura->codigoPostal )) {
			$domicilioReceptor ["codigoPostal"] = $dataFactura->codigoPostalR;
		}

		if (! empty ( $dataFactura->coloniaR )) {
			$domicilioReceptor ["colonia"] = $dataFactura->coloniaR;
		}

		if (! empty ( $dataFactura->estadoR )) {
			$domicilioReceptor ["estado"] = $dataFactura->estadoR;
		}

		if (! empty ( $dataFactura->localidadR )) {
			$domicilioReceptor ["localidad"] = $dataFactura->localidadR;
		}

		if (! empty ( $dataFactura->municipioR )) {
			$domicilioReceptor ["municipio"] = $dataFactura->municipioR;
		}

		if (! empty ( $dataFactura->nroExteriorR )) {
			$domicilioReceptor ["noExterior"] = $dataFactura->nroExteriorR;
		}

		if (! empty ( $dataFactura->nroInteriorR )) {
			$domicilioReceptor ["noInterior"] = $dataFactura->nroInteriorR;
		}

		if (! empty ( $dataFactura->paisR )) {
			$domicilioReceptor ["pais"] = $dataFactura->paisR;
		}

		$root->addChild ( 'cfdi:Receptor', true, [
				'nombre' => $dataFactura->nmbRecep,
				'rfc' => $dataFactura->rFCRecep
		] )->addChild ( 'cfdi:Domicilio', '', $domicilioReceptor );

		$conceptosNode = $root->addChild ( 'cfdi:Conceptos', true );

		foreach ( $dataFactura->conceptos as $concepto ) {

			$cantidadItem = round ( $concepto->qtyItem, 2 );

			$conceptosNode->addChild ( 'cfdi:Concepto', '', [
					'cantidad' => $cantidadItem,
					'descripcion' => $concepto->dscItem,
					'importe' => Utils::formatNumber ( $concepto->prcNetoItem, $dataFactura->precission ),
					'noIdentificacion' => $concepto->cdgItem,
					'unidad' => $concepto->unmdItem,
					'valorUnitario' => Utils::formatNumber ( $concepto->montoNetoItem, $dataFactura->precission )
			] );
		}

		$totalesImpuestos = array ();
		//Nodo de impuestos retenidos
		if (count ( $impuestosRetenidos ) > 0) {
			$totalesImpuestos ['totalImpuestosRetenidos'] = Utils::formatNumber ( $totalImpuestosRetenidos, $dataFactura->precission );
		}

		// Nodo de impuestos trasladados
		$totalesImpuestos ['totalImpuestosTrasladados'] = Utils::formatNumber ( $totalImpuestosTraslados, $dataFactura->precission );

		$nodoImpuestos = $root->addChild ( 'cfdi:Impuestos', true, $totalesImpuestos );


		if (count ( $impuestosRetenidos ) > 0) {

			$nodoRetenciones = $nodoImpuestos->addChild ( 'cfdi:Retenciones', true );

			foreach ( $impuestosRetenidos as $retencion ) {

				$nodoRetenciones->addChild ( 'cfdi:Retencion', '', [
						'importe' => Utils::formatNumber ( $retencion->montoImp, $dataFactura->precission ),
						'impuesto' => trim ( $retencion->tipoImp )
				] );
			}
		}


		if (count ( $impuestosTrasladados ) > 0) {

			$nodoTraslados = $nodoImpuestos->addChild ( 'cfdi:Traslados', true );

			foreach ( $impuestosTrasladados as $traslado ) {

				$nodoTraslados->addChild ( 'cfdi:Traslado', '', [
						'importe' => Utils::formatNumber ( $traslado->montoImp, $dataFactura->precission ),
						'impuesto' => trim ( $traslado->tipoImp ),
						'tasa' => Utils::formatNumber ( $traslado->tasaImp, $dataFactura->precission )
				] );
			}
		}



		///Timbre fiscal
		$fechaTimbrado = new \DateTime($dataFactura->fechaTimbrado);
		$fechaTimbrado = date_format ( $fechaTimbrado, 'Y-m-d\TH:i:s' );

		if($dataFactura->milisegundos > 0){
			$fechaTimbrado .= ".".str_pad ( $dataFactura->milisegundos, 3, "0", STR_PAD_LEFT );
		}

		$nodoComplemento = $root->addChild ( 'cfdi:Complemento', true );

		if(count($impuestosRetendosLocales) > 0 && count($impuestosTrasladadosLocales) > 0){

			$nodoImpuestosLocales = $nodoComplemento->addChild('implocal:ImpuestosLocales', true, ['TotaldeRetenciones' => $totalImpuestosRetenidosLocales, 'TotaldeTraslados' => $totalImpuestosTrasladosLocales, 'version' => '1.0']);

			if(count($impuestosRetendosLocales) > 0){
				foreach ($impuestosRetendosLocales as $impuestoRetLocal){
					$nodoImpuestosLocales->addChild('implocal:RetencionesLocales', '', ['ImpLocRetenido' => $impuestoRetLocal->tipoImp, 'Importe' => Utils::formatNumber($impuestoRetLocal->montoImp, 2), 'TasadeRetencion' => Utils::formatNumber($impuestoRetLocal->tasaImp, $dataFactura->precission)]);
				}
			}

			if(count($impuestosTrasladadosLocales) > 0){
				foreach ($impuestosTrasladadosLocales as $impuestoTrasLocal){
					$nodoImpuestosLocales->addChild('implocal:TrasladosLocales', '', ['ImpLocTrasladado' => $impuestoTrasLocal->tipoImp, 'Importe' => Utils::formatNumber($impuestoTrasLocal->montoImp, 2), 'TasadeTraslado' => Utils::formatNumber($impuestoTrasLocal->tasaImp, $dataFactura->precission)]);
				}
			}

		}


		$nodoComplemento->addChild ( 'tfd:TimbreFiscalDigital', '', [
				'FechaTimbrado' => $fechaTimbrado,
				"xsi:schemaLocation" => "http://www.sat.gob.mx/TimbreFiscalDigital http://www.sat.gob.mx/sitio_internet/timbrefiscaldigital/TimbreFiscalDigital.xsd",
				'UUID' => $dataFactura->uUID,
				'noCertificadoSAT' => $dataFactura->noCertificadoSAT,
				'selloCFD' => $dataFactura->selloCFD,
				'selloSAT' => $dataFactura->selloSAT,
				'version' => '1.0'
		]
		 );

		return $xml;
	}







	public static function createXML33($dataFactura, $proveedor) {


		$cfdiNS = new FluidNamespace ( 'cfdi', 'http://www.sat.gob.mx/cfd/3' );
		$xsiNS = new FluidNamespace ( 'xsi', 'http://www.w3.org/2001/XMLSchema-instance' );
		$xsiSLNS = new FluidNamespace ( 'schemaLocation', 'http://www.sat.gob.mx/sitio_internet/cfd/3/cfdv33.xsd' );
		$tfdNs = new FluidNamespace ( 'tfd', 'http://www.sat.gob.mx/TimbreFiscalDigital' );
		$pagosNs = new FluidNamespace ( 'pago10', 'http://www.sat.gob.mx/Pagos' );
		$impLNs = new FluidNamespace ( 'implocal', 'http://www.sat.gob.mx/implocal' );
		$nomLNs = new FluidNamespace ( 'nomina12', 'http://www.sat.gob.mx/nomina12' );
		$cceLNs = new FluidNamespace ( 'cce11', 'http://www.sat.gob.mx/ComercioExterior11' );
        $fondoNs = new FluidNamespace ( 'FondoGarantia', 'http://201.148.157.232:81/Content/FondoGarantia' );
        $ieduNs = new FluidNamespace ( 'iedu', 'http://www.sat.gob.mx/iedu' );

		$xml = new FluidXml ( null ); // A document without a root node.
		$xml->namespace ( $cfdiNS ); // Let's register the namespaces associated with our document.
		$xml->namespace ( $xsiNS );
		$xml->namespace ( $tfdNs );
		$xml->namespace ( $pagosNs );

		if(isset($dataFactura->addenda) && $dataFactura->addenda != null){
            $xml->namespace ( $fondoNs );
        }

		if($dataFactura->tipoDeComprobante != 'P'){
		    $xml->namespace ( $nomLNs );
		    $xml->namespace ( $impLNs );
		    $xml->namespace ( $cceLNs );
		}

        $listNameSpaces = "http://www.sat.gob.mx/cfd/3 http://www.sat.gob.mx/sitio_internet/cfd/3/cfdv33.xsd http://www.sat.gob.mx/Pagos http://www.sat.gob.mx/sitio_internet/cfd/Pagos/Pagos10.xsd http://www.sat.gob.mx/implocal http://www.sat.gob.mx/sitio_internet/cfd/implocal/implocal.xsd http://www.sat.gob.mx/nomina12 http://www.sat.gob.mx/sitio_internet/cfd/nomina/nomina12.xsd http://www.sat.gob.mx/ComercioExterior11 http://www.sat.gob.mx/sitio_internet/cfd/ComercioExterior11/ComercioExterior11.xsd";

        if(isset($dataFactura->conceptos[0]->alumno) && $dataFactura->conceptos[0]->alumno != null){
            $xml->namespace ( $ieduNs );
            $listNameSpaces .= ' http://www.sat.gob.mx/iedu http://www.sat.gob.mx/sitio_internet/cfd/iedu/iedu.xsd';
        }

		$root = $xml->addChild ( 'cfdi:Comprobante', true );

		$rootNode = $root [0];

		// Agrenado namespaces
		$rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$cfdiNS->id()}", $cfdiNS->uri () );
		$rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$xsiNS->id()}", $xsiNS->uri () );
		$rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$pagosNs->id()}", $pagosNs->uri () );

		if($dataFactura->tipoDeComprobante != 'P'){
		    $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$impLNs->id()}", $impLNs->uri () );
		    $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$nomLNs->id()}", $nomLNs->uri () );
		    $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$cceLNs->id()}", $cceLNs->uri () );
		}

		$rootNode->setAttribute ( 'xsi:schemaLocation', $listNameSpaces );

		$fechaEmision = date_format ( new \DateTime ( $dataFactura->timeStamp ), 'Y-m-d\TH:i:s' );

		// Creando el nodo Comprobante
		$totalDocumento = 0;
		$subTotalDocumento = 0;
		if($dataFactura->tipoDeComprobante != 'P'){
		    $totalDocumento = Utils::formatNumber ( $dataFactura->total, $dataFactura->precission );
		    $subTotalDocumento = Utils::formatNumber ( $dataFactura->subtotal, $dataFactura->precission );
		}


		$rootNode->setAttribute ( 'Version', '3.3' );
		$rootNode->setAttribute ( 'Serie', $dataFactura->serie );
		$rootNode->setAttribute ( 'Folio', $dataFactura->folio );
		$rootNode->setAttribute ( 'Fecha', $fechaEmision );
		$rootNode->setAttribute ( 'NoCertificado', $dataFactura->noCertificado );
		$rootNode->setAttribute ( 'Certificado', $dataFactura->certificado );
		$rootNode->setAttribute ( 'SubTotal', $subTotalDocumento);
		$rootNode->setAttribute ( 'Moneda', $dataFactura->moneda );
		$rootNode->setAttribute ( 'Total',  $totalDocumento);
		$rootNode->setAttribute ( 'TipoDeComprobante', $dataFactura->tipoDeComprobante );

		if($dataFactura->formaPago != '')
			$rootNode->setAttribute ( 'FormaPago', $dataFactura->formaPago );

		if($dataFactura->metodoDePago != '')
			$rootNode->setAttribute ( 'MetodoPago', $dataFactura->metodoDePago );

		if($dataFactura->condicionesDePago != "")
			$rootNode->setAttribute ( 'CondicionesDePago', $dataFactura->condicionesDePago );


		if ($dataFactura->descuento > 0) {
			$montoDescuento = Utils::formatNumber ( $dataFactura->descuento, $dataFactura->precission );
			$rootNode->setAttribute ( 'Descuento', $montoDescuento );
		}

		if($dataFactura->moneda != 'MXN' && $dataFactura->moneda != 'XXX')
			$rootNode->setAttribute ( 'TipoCambio', Utils::formatNumber ( $dataFactura->tipoCambio, $dataFactura->precission ) );

		$rootNode->setAttribute ( 'LugarExpedicion', $dataFactura->lugarExpedicion );



		if(count($dataFactura->facturasRelacionadas) > 0){

			$nodoRelacionados = $root->addChild ( 'cfdi:CfdiRelacionados', true, [
				'TipoRelacion' => $dataFactura->tipoFacturasRelacionadas
			]);

			foreach ($dataFactura->facturasRelacionadas as $facturaRelacionada){

				$nodoRelacionados->addChild ( 'cfdi:CfdiRelacionado', true, [
						'UUID' => $facturaRelacionada->uuid
				]);

			}
		}


		$emisorObj = [
			'Nombre' => $dataFactura->nmbEmisor,
			'Rfc' => $dataFactura->rFCEmisor,
			'RegimenFiscal' => $dataFactura->regimenFiscal];

		/*
		if($dataFactura->tipoPersona ==  2 && isset($dataFactura->curp) && $dataFactura->curp != ''){
			$emisorObj['Curp'] = $dataFactura->curp;
		}
		*/

		$root->addChild ( 'cfdi:Emisor', true, $emisorObj);

		$recepArray = [
				'Nombre' => $dataFactura->nmbRecep,
				'Rfc' => $dataFactura->rFCRecep,
				'UsoCFDI' => $dataFactura->usoCFDI
		];

		if($dataFactura->numRegIdTrib != '')
			$recepArray['NumRegIdTrib'] = $dataFactura->numRegIdTrib;

		if($dataFactura->pais != '')
			$recepArray['ResidenciaFiscal'] = $dataFactura->pais;

        if($dataFactura->numRegIdTrib == '881373697')
            $recepArray['ResidenciaFiscal'] = "USA";

		$root->addChild ( 'cfdi:Receptor', true, $recepArray);

		$conceptosNode = $root->addChild ( 'cfdi:Conceptos', true );

		foreach ( $dataFactura->conceptos as $concepto ) {

			$agregarImpuestos = '';

			$importe = 0;
			$valUnit = 0;
			if($dataFactura->tipoDeComprobante != 'P'){
			    $importe = Utils::formatNumber ( $concepto->prcNetoItem, $dataFactura->precission );
			    $valUnit = Utils::formatNumber ( $concepto->montoNetoItem, $dataFactura->precission );
			}

			$attrConcepto = [
					'Cantidad' => $concepto->qtyItem,
					'ClaveProdServ' => $concepto->clave,
					'ClaveUnidad' => $concepto->unmdItem,
					'Descripcion' => $concepto->dscItem,
			         'Importe' => $importe,
			         'ValorUnitario' => $valUnit
			];

			if($concepto->cdgItem != '')
				$attrConcepto['NoIdentificacion'] = $concepto->cdgItem;

            if(!empty($concepto->unmdItemDesc))
                $attrConcepto['Unidad'] = $concepto->unmdItemDesc;

			if($concepto->descuento > 0)
				$attrConcepto['Descuento'] = Utils::formatNumber ($concepto->descuento, $dataFactura->precission);




			$nodoConcepto = $conceptosNode->addChild ( 'cfdi:Concepto', true, $attrConcepto);

			if(count($concepto->impuestosTrasladados) > 0 || count($concepto->impuestosRetenidos) > 0){

				$impuestosNode = $nodoConcepto->addChild ( 'cfdi:Impuestos', true );

				//Impuestos traslados
				if(count($concepto->impuestosTrasladados) > 0){

					$trasladosNode = $impuestosNode->addChild ( 'cfdi:Traslados', true );

					foreach($concepto->impuestosTrasladados as $impTraslado){

						$atributos = [
								'Base' => Utils::formatNumber ( $impTraslado->base, $dataFactura->precission ),
								'Impuesto' => $impTraslado->tipoImp,
								'TipoFactor' => $impTraslado->factor
						];

						if($impTraslado->factor != 'Exento'){
							$atributos['TasaOCuota'] = Utils::formatNumber ( $impTraslado->tasaImp, 6 );
							$atributos['Importe'] = Utils::formatNumber ($impTraslado->montoImp,2);
						}

						$trasladosNode->addChild ( 'cfdi:Traslado', true, $atributos);

					}
				}

				//Impuestos retenidos
				if(count($concepto->impuestosRetenidos) > 0){

					$retencionesNode = $impuestosNode->addChild ( 'cfdi:Retenciones', true );

					foreach($concepto->impuestosRetenidos as $impRetenido){

						$attributos = [
								'Base' => Utils::formatNumber ( $impRetenido->base, $dataFactura->precission ),
								'Impuesto' => $impRetenido->tipoImp,
								'TipoFactor' => $impRetenido->factor
						];

						if($impRetenido->factor != 'Exento'){
							$attributos['TasaOCuota'] = Utils::formatNumber ( $impRetenido->tasaImp, 6 );
							$attributos['Importe'] = Utils::formatNumber ( $impRetenido->montoImp, $dataFactura->precission );
						}

						$retencionesNode->addChild ( 'cfdi:Retencion', true, $attributos);
					}
				}

			}

			if(isset($dataFactura->cuentaPredial) && $dataFactura->cuentaPredial != ''){
				$nodoConcepto->addChild ( 'cfdi:CuentaPredial', true, ['Numero' => $dataFactura->cuentaPredial] );
			}

		}


		if(count($dataFactura->impuestosTrasladados) > 0 || count($dataFactura->impuestosRetenidos) > 0){

			$impuestosGeneralesNode = $root->addChild ( 'cfdi:Impuestos', true);

			if(count($dataFactura->impuestosRetenidos) > 0){

				$impuestosGeneralesNode->setAttribute ( 'TotalImpuestosRetenidos', Utils::formatNumber ( $dataFactura->totalImpRetenidos, $dataFactura->precission ) );

				$retencionesGralNode = $impuestosGeneralesNode->addChild ( 'cfdi:Retenciones', true );

				foreach ($dataFactura->impuestosRetenidos as $impRetencion){
					$retencionesGralNode->addChild ( 'cfdi:Retencion', true, [
							'Impuesto' => $impRetencion->impuesto,
							'Importe' => Utils::formatNumber ( $impRetencion->importe, $dataFactura->precission )
					]);
				}

			}

			if(count($dataFactura->impuestosTrasladados) > 0){

				$impuestosGeneralesNode->setAttribute ( 'TotalImpuestosTrasladados', Utils::formatNumber ( $dataFactura->totalImpTrasladados, $dataFactura->precission ) );

				$retencionesGralNode = $impuestosGeneralesNode->addChild ( 'cfdi:Traslados', true );

				foreach ($dataFactura->impuestosTrasladados as $impTraslado){
					$retencionesGralNode->addChild ( 'cfdi:Traslado', true, [
							'Impuesto' => $impTraslado->impuesto,
							'Importe' => Utils::formatNumber ( $impTraslado->importe, $dataFactura->precission ),
							'TipoFactor' => $impTraslado->tipoFactor,
							'TasaOCuota' => Utils::formatNumber ( $impTraslado->tasaOCuota , 6 )
					]);
				}
			}


		}

		if(isset($dataFactura->sello) ||
				(isset($dataFactura->pagos) && count($dataFactura->pagos) > 0) ||
					(count($dataFactura->impuestosRetenidosLcl) > 0 || count($dataFactura->impuestosTrasladadosLcl) > 0) || isset($dataFactura->nomina)
				|| (isset($dataFactura->comercio) && $dataFactura->comercio != null) ) {


			$nodoComplemento = $root->addChild ( 'cfdi:Complemento', true );


			if( (isset($dataFactura->impuestosRetenidosLcl) && count($dataFactura->impuestosRetenidosLcl)) > 0
					|| (isset($dataFactura->impuestosTrasladadosLcl) && count($dataFactura->impuestosTrasladadosLcl) > 0) ){

				$properties = array('version' => '1.0');
				$properties['TotaldeRetenciones'] = Utils::formatNumber ( $dataFactura->impuestosRetenidosLclTotal, $dataFactura->precission );
				$properties['TotaldeTraslados'] = Utils::formatNumber ( $dataFactura->impuestosTrasladadosLclTotal, $dataFactura->precission );

				$nodoImpuestosLocales = $nodoComplemento->addChild('implocal:ImpuestosLocales', true, $properties);

				if(isset($dataFactura->impuestosRetenidosLcl) && count($dataFactura->impuestosRetenidosLcl) > 0){
					foreach ($dataFactura->impuestosRetenidosLcl as $impuestoRetLocal){
						$nodoImpuestosLocales->addChild('implocal:RetencionesLocales', '',
								['ImpLocRetenido' => $impuestoRetLocal->tipoImp,
										'Importe' => Utils::formatNumber($impuestoRetLocal->montoImp, 2),
										'TasadeRetencion' => Utils::formatNumber($impuestoRetLocal->tasaImp, $dataFactura->precission)]);
					}
				}

				if(isset($dataFactura->impuestosTrasladadosLcl) && count($dataFactura->impuestosTrasladadosLcl) > 0){
					foreach ($dataFactura->impuestosTrasladadosLcl as $impuestoTrasLocal){
						$nodoImpuestosLocales->addChild('implocal:TrasladosLocales', '', ['ImpLocTrasladado' => $impuestoTrasLocal->tipoImp, 'Importe' => Utils::formatNumber($impuestoTrasLocal->montoImp, 2), 'TasadeTraslado' => Utils::formatNumber($impuestoTrasLocal->tasaImp, $dataFactura->precission)]);
					}
				}

			}


			if(isset($dataFactura->pagos) && count($dataFactura->pagos) > 0){

				$nodoPagos = $nodoComplemento->addChild ( 'pago10:Pagos', true, [
						'Version' => '1.0'
				]);

				foreach ($dataFactura->pagos as $pago){

					$fechaPago = new \DateTime ( $pago->FechaPago );
					$fechaPago = date_format ( $fechaPago, 'Y-m-d\TH:i:s' );

					$pagoElment = [
					    'FechaPago' => $fechaPago,
					    'FormaDePagoP' => $pago->FormaDePagoP,
					    'MonedaP' => $pago->MonedaP,
					    'Monto' => Utils::formatNumber ( $pago->Monto, 2 )
					];

					if($pago->banco != ''){
					    $pagoElment['RfcEmisorCtaOrd'] = $pago->banco;
					}

					if($pago->cuenta != ''){
					    $pagoElment['CtaOrdenante'] = $pago->cuenta;
					}

                    if(isset($pago->rfcEmisorCtaBen) && $pago->rfcEmisorCtaBen != null && $pago->rfcEmisorCtaBen != ''){
                        $pagoElment['RfcEmisorCtaBen'] = $pago->rfcEmisorCtaBen;
                    }

                    if(isset($pago->ctaBeneficiario) && $pago->ctaBeneficiario != null && $pago->ctaBeneficiario != ''){
                        $pagoElment['CtaBeneficiario'] = $pago->ctaBeneficiario;
                    }

                    if(isset($pago->noOperacion) && $pago->noOperacion != ''){
                        $pagoElment['NumOperacion'] = $pago->noOperacion;
                    }

                    if($pago->TipoCambioP > 1){
                        $pagoElment['TipoCambioP'] = $pago->TipoCambioP;
                    }

					$nodoPago = $nodoPagos->addChild ( 'pago10:Pago', true, $pagoElment);

					if(count($pago->documentosRelacionado) > 0)
					foreach ($pago->documentosRelacionado as $doctoRelacionado){

                        $docRelData = [
                            'IdDocumento' => $doctoRelacionado->IdDocumento,
                            'MonedaDR' => $doctoRelacionado->MonedaDR,
                            'MetodoDePagoDR' => $doctoRelacionado->MetodoDePagoDR,
                            'NumParcialidad' => $doctoRelacionado->NumParcialidad,
                            'ImpSaldoAnt' => Utils::formatNumber ( $doctoRelacionado->ImpSaldoAnt, 2 ),
                            'ImpPagado' => Utils::formatNumber ( $doctoRelacionado->ImpPagado, 2 ),
                            'ImpSaldoInsoluto' => Utils::formatNumber ( $doctoRelacionado->ImpSaldoInsoluto, 2 )
                        ];

                        if($doctoRelacionado->TipoCambioDR != null)
                            $docRelData['TipoCambioDR'] = Utils::formatNumber ( $doctoRelacionado->TipoCambioDR, 2 );

                        $nodoPago->addChild ( 'pago10:DoctoRelacionado', '', $docRelData);
					}
				}
			}


			if(isset($dataFactura->nomina)){

				$nomina = $dataFactura->nomina;

				$receptor = $nomina['Receptor'];

				$arrayNomina = [
						'Version' => $nomina['Version'],
						'FechaPago' => $nomina['FechaPago'],
						'FechaInicialPago' => $nomina['FechaInicialPago'],
						'FechaFinalPago' => $nomina['FechaFinalPago'],
						'NumDiasPagados' => $nomina['NumDiasPagados'],
						'TipoNomina' => $nomina['TipoNomina']
				];


                $OtrosPagos = $nomina['OtrosPagos'];

                if($nomina['TotalDeducciones'] > 0)
                    $arrayNomina['TotalDeducciones'] = $nomina['TotalDeducciones'];

                if($nomina['TotalPercepciones'] > 0)
                    $arrayNomina['TotalPercepciones'] = $nomina['TotalPercepciones'];

                if(count($OtrosPagos['listadoOtrosPagos']) > 0)
                    $arrayNomina['TotalOtrosPagos'] = $nomina['TotalOtrosPagos'];


				$nodoNomina = $nodoComplemento->addChild ( 'nomina12:Nomina', true, $arrayNomina);

				$emisorNomina = array();

				if(isset($receptor ['TipoContrato']) && $receptor ['TipoContrato'] != '09' && $receptor ['TipoContrato'] != '10' && $receptor ['TipoContrato'] != '99'){
					$emisorNomina = ['RegistroPatronal' => $nomina['Emisor']['RegistroPatronal']];
				}

				if($dataFactura->tipoPersona ==  2 && isset($dataFactura->curp) && $dataFactura->curp != ''){
					$emisorNomina['Curp'] = $dataFactura->curp;
				}

				$nodoNomina->addChild ( 'nomina12:Emisor', true, $emisorNomina);

				$arrayData = [];

				if (! empty ( $receptor ['Curp'] ))
					$arrayData ['Curp'] = $receptor ['Curp'];

				if (! empty ( $receptor ['TipoContrato'] ))
					$arrayData ['TipoContrato'] = trim ( $receptor ['TipoContrato'] );

				if (! empty ( $receptor ['TipoRegimen'] )) {
					$arrayData ['TipoRegimen'] = $receptor ['TipoRegimen'];
				}

				if (! empty ( $receptor ['NumEmpleado'] ))
					$arrayData ['NumEmpleado'] = $receptor ['NumEmpleado'];

				if (! empty ( $receptor ['PeriodicidadPago'] ))
					$arrayData ['PeriodicidadPago'] = $receptor ['PeriodicidadPago'];

				if (! empty ( $receptor ['ClaveEntFed'] ))
					$arrayData ['ClaveEntFed'] = trim ( $receptor ['ClaveEntFed'] );

				if (isset ($receptor ['NumSeguridadSocial'])
                    &&  ! empty ( $receptor ['NumSeguridadSocial'] )
                    && (!isset($receptor ['TipoRegimen']) || $receptor ['TipoRegimen'] != 10))
					$arrayData ['NumSeguridadSocial'] = trim ( $receptor ['NumSeguridadSocial'] );

				if (! empty ( $receptor ['FechaInicioRelLaboral'] ) && $receptor ['TipoRegimen'] != 10)
					$arrayData ['FechaInicioRelLaboral'] = trim ( $receptor ['FechaInicioRelLaboral'] );

				if (! empty ($receptor ['Antigüedad'] ) && $receptor ['TipoRegimen'] != 10)
					$arrayData ['Antigüedad'] = $receptor['Antigüedad'];

				if (! empty ($receptor ['Puesto'] ))
					$arrayData ['Puesto'] = trim ( $receptor ['Puesto'] );

                if (! empty ( $receptor ['SalarioBaseCotApor'] ) && $receptor ['SalarioBaseCotApor'] > 0)
                    $arrayData ['SalarioBaseCotApor'] = $receptor ['SalarioBaseCotApor'];

                if (! empty ( $receptor ['SalarioDiarioIntegrado'] ) && $receptor ['SalarioDiarioIntegrado'] > 0 && $receptor ['TipoRegimen'] != 10)
                    $arrayData ['SalarioDiarioIntegrado'] = $receptor ['SalarioDiarioIntegrado'];

                if (! empty ( $receptor ['RiesgoPuesto'] ) && $receptor ['TipoRegimen'] != 10) {
                    $arrayData ['RiesgoPuesto'] = $receptor ['RiesgoPuesto'];
                }


				$nodoNomina->addChild ( 'nomina12:Receptor', true, $arrayData);


				$percepciones = $nomina['Percepciones'];
				$deducciones = $nomina['Deducciones'];
				$OtrosPagos = $nomina['OtrosPagos'];
				$Incapacidades = $nomina['Incapacidades'];


				if(count($percepciones['listadoPercepciones']) > 0){

					$totalesPercepciones = [];

					$totalesPercepciones['TotalGravado'] = Utils::formatNumber($percepciones['totalPercepcionGravado']);

					$totalesPercepciones['TotalExento'] = Utils::formatNumber($percepciones['totalPercepcionExcento']);

					if($percepciones['totalSueldos'] > 0){
						$totalesPercepciones['TotalSueldos'] = Utils::formatNumber($percepciones['totalSueldos']);
					}

					if($percepciones['totalSeparacionIndemnizacion'] > 0){
						$totalesPercepciones['TotalSeparacionIndemnizacion'] = Utils::formatNumber($percepciones['totalSeparacionIndemnizacion']);
					}

					if($percepciones['totalJubilacionPension'] > 0){
						$totalesPercepciones['TotalJubilacionPensionRetiro'] = Utils::formatNumber($percepciones['totalJubilacionPension']);
					}


					$nodoPercepciones = $nodoNomina->addChild ( 'nomina12:Percepciones', true, $totalesPercepciones);

					foreach ($percepciones['listadoPercepciones'] as $percepcionObj){


						$percepcionElement = array('TipoPercepcion' => $percepcionObj['TipoPercepcion'], 'Clave' => $percepcionObj['Clave'],
								'Concepto' => $percepcionObj['Concepto'], 'ImporteGravado' => $percepcionObj['ImporteGravado'],
								'ImporteExento' => $percepcionObj['ImporteExento']);


						$nodoPercepcion = $nodoPercepciones->addChild ( 'nomina12:Percepcion', true, $percepcionElement);

						if($percepcionObj['TipoPercepcion'] == '019'){
							$horaExtra = array('Dias' => $percepcionObj['Dias'], 'TipoHoras' => str_pad ( $percepcionObj['TipoHoras'], 2, "0", STR_PAD_LEFT ),
									'HorasExtra' => $percepcionObj['HorasExtra'], 'ImportePagado'  => $percepcionObj['ImportePagado']);

							$nodoPercepcion->addChild ( 'nomina12:HorasExtra', true, $horaExtra);
						}

					}

					if($dataFactura->liquidacion != null){

					    $separacionIndemnizacionElement = array(
					        'TotalPagado' => Utils::formatNumber($dataFactura->liquidacion->totalPagado),
                            'NumAñosServicio' => Utils::formatNumber($dataFactura->liquidacion->ultimoSueldoMensOrd),
                            'UltimoSueldoMensOrd' => Utils::formatNumber($dataFactura->liquidacion->numAniosServicio),
                            'IngresoAcumulable' => Utils::formatNumber($dataFactura->liquidacion->ingresoAcumulable),
                            'IngresoNoAcumulable' => Utils::formatNumber($dataFactura->liquidacion->ingresoNoAcumulable));

                        $nodoPercepciones->addChild ( 'nomina12:SeparacionIndemnizacion', true, $separacionIndemnizacionElement);

                    }

				}


				if(count($deducciones['listadoDeducciones']) > 0){

					$totalesDeducciones = [];

					if($deducciones['totalOtrasDeducciones'] > 0){
						$totalesDeducciones['TotalOtrasDeducciones'] = Utils::formatNumber($deducciones['totalOtrasDeducciones']);
					}

					if($deducciones['totalImpuestosRetenidos'] > 0){
						$totalesDeducciones['TotalImpuestosRetenidos'] = Utils::formatNumber($deducciones['totalImpuestosRetenidos']);
					}

					$nodoDeducciones = $nodoNomina->addChild ( 'nomina12:Deducciones', true, $totalesDeducciones);

					foreach ($deducciones['listadoDeducciones'] as $deduccionObj){
						$nodoDeducciones->addChild ( 'nomina12:Deduccion', true, $deduccionObj);
					}

				}


				if(count($OtrosPagos['listadoOtrosPagos']) > 0){

					$nodoDeducciones = $nodoNomina->addChild ( 'nomina12:OtrosPagos', true);

					foreach ($OtrosPagos['listadoOtrosPagos'] as $otroPago){

						$nodoOtroPago = $nodoDeducciones->addChild ( 'nomina12:OtroPago', true, $otroPago);

						if($otroPago['TipoOtroPago'] == '002'){
							$nodoOtroPago->addChild ( 'nomina12:SubsidioAlEmpleo', true, array('SubsidioCausado' => $otroPago['Importe']));
						}

					}

				}

				if(count($Incapacidades['listadoIncapacidades']) > 0){

					$nodoDeducciones = $nodoNomina->addChild ( 'nomina12:Incapacidades', true);

					foreach ($Incapacidades['listadoIncapacidades'] as $incapacidad){
						$nodoDeducciones->addChild ( 'nomina12:Incapacidad', true, $incapacidad);
					}

				}


			}

			//Complemento comercio exterior
			if((isset($dataFactura->comercio) && $dataFactura->comercio != null)){

				$comercio = $dataFactura->comercio;

				$properties = array('Version' => '1.1');
				$properties['ClaveDePedimento'] = $comercio->datos->clavePedimento;
				$properties['Incoterm'] = $comercio->datos->Incoterm;
				$properties['TipoOperacion'] = $comercio->datos->tipoOperacion;
				$properties['CertificadoOrigen'] = 0;
				$properties['Subdivision'] = 0;


				if($dataFactura->moneda == 'USD' || $dataFactura->moneda == 'EUR'){
					$properties['TipoCambioUSD'] = Utils::formatNumber ( $dataFactura->tipoCambio, $dataFactura->precission );
					$properties['TotalUSD'] = Utils::formatNumber ( $dataFactura->total, $dataFactura->precission );
				}else{
				    if($properties['TipoOperacion'] == 2) {
                        $properties['TipoCambioUSD'] = Utils::formatNumber(0);
                        $properties['TotalUSD'] = Utils::formatNumber(0);
                    }
                }

				$nodoComercioE = $nodoComplemento->addChild('cce11:ComercioExterior', true, $properties);

				$datosEmisorCE = array();
				if($dataFactura->tipoPersona ==  2 && isset($comercio->datos->curp) && $comercio->datos->curp != '')
					$datosEmisorCE['Curp'] = $comercio->datos->curp;

				$nodoEmisorComercioE = $nodoComercioE->addChild('cce11:Emisor', true, $datosEmisorCE);

				$emisorCE = $comercio->emisor;
				$domicilioEmisor = array(
						'Calle' => $emisorCE->calle,
						'Estado' => $emisorCE->estado,
						'Pais' => $emisorCE->pais,
						'CodigoPostal' => $emisorCE->codigoPostal
				);

				if(!empty($emisorCE->nroExterior))
					$domicilioEmisor['NumeroExterior'] = $emisorCE->nroExterior;

				if(!empty($emisorCE->nroInterior))
					$domicilioEmisor['NumeroInterior'] = $emisorCE->nroInterior;

				if(!empty($emisorCE->colonia))
					$domicilioEmisor['Colonia'] = $emisorCE->colonia;

				if(!empty($emisorCE->municipio))
					$domicilioEmisor['Municipio'] = $emisorCE->municipio;

				$nodoEmisorComercioE->addChild('cce11:Domicilio', true, $domicilioEmisor);


				$nodoReceptorComercioE = $nodoComercioE->addChild('cce11:Receptor', true);

				$receptorCE = $comercio->receptor;
				$domicilioReceptor = array(
						'Calle' => $receptorCE->calle,
						'Estado' => $receptorCE->estado,
						'Pais' => $receptorCE->pais,
						'CodigoPostal' => $receptorCE->codigoPostal
				);

				if(!empty($receptorCE->nroExterior))
					$domicilioReceptor['NumeroExterior'] = $receptorCE->nroExterior;

				if(!empty($receptorCE->colonia))
					$domicilioReceptor['Colonia'] = $receptorCE->colonia;

				if(!empty($receptorCE->nroInterior))
					$domicilioReceptor['NumeroInterior'] = $receptorCE->nroInterior;

				if(!empty($receptorCE->municipio))
					$domicilioReceptor['Municipio'] = $receptorCE->municipio;

				$nodoReceptorComercioE->addChild('cce11:Domicilio', true, $domicilioReceptor);



				$nodoDestinatarioComercioE = $nodoComercioE->addChild('cce11:Destinatario', true, ['NumRegIdTrib' =>  $comercio->datos->rfcExtranjero]);
				$nodoDestinatarioComercioE->addChild('cce11:Domicilio', true, $domicilioReceptor);


				$nodoMercanciasComercioE = $nodoComercioE->addChild('cce11:Mercancias', true);

				foreach ( $dataFactura->conceptos as $concepto ) {



					$conceptoComercio = [
						'CantidadAduana' => $concepto->qtyItem
					];

					if(!empty($concepto->cdgItem))
						$conceptoComercio['NoIdentificacion'] = $concepto->cdgItem;

					if(!empty($concepto->fracArancelaria)){

						$fraccion = FraccionArancelaria::where('codigo','=', $concepto->fracArancelaria)->first();
						$conceptoComercio['FraccionArancelaria'] = $concepto->fracArancelaria;
						$conceptoComercio['UnidadAduana'] = $fraccion->unidad;
					}


					if($dataFactura->moneda == 'USD'){
						$conceptoComercio['ValorDolares'] = Utils::formatNumber ( $concepto->prcNetoItem, $dataFactura->precission );
						$conceptoComercio['ValorUnitarioAduana'] = Utils::formatNumber ( $concepto->montoNetoItem, $dataFactura->precission );
					}


					$nodoMercanciasComercioE->addChild('cce11:Mercancia', true, $conceptoComercio);
				}



			}




			if(isset($dataFactura->sello) && $dataFactura->sello != ''){


				$rootNode->setAttribute ( 'Sello', $dataFactura->sello );

				///Timbre fiscal
				$fechaTimbrado = new \DateTime ( $dataFactura->fechaTimbrado );
				$fechaTimbrado = date_format ( $fechaTimbrado, 'Y-m-d\TH:i:s' );

				if($dataFactura->milisegundos > 0){
					$fechaTimbrado .= ".".str_pad ( $dataFactura->milisegundos, 3, "0", STR_PAD_LEFT );
				}


				$nodoComplemento->addChild ( 'tfd:TimbreFiscalDigital', '', [
						'FechaTimbrado' => $fechaTimbrado,
						"xsi:schemaLocation" => "http://www.sat.gob.mx/TimbreFiscalDigital http://www.sat.gob.mx/sitio_internet/cfd/TimbreFiscalDigital/TimbreFiscalDigitalv11.xsd",
						'UUID' => $dataFactura->uUID,
						'NoCertificadoSAT' => $dataFactura->noCertificadoSAT,
						'SelloCFD' => $dataFactura->selloCFD,
						'SelloSAT' => $dataFactura->selloSAT,
						'RfcProvCertif' => $dataFactura->rfcProvCertif,
						'Version' => '1.1',
				]);

			}
		}

		if(isset($dataFactura->addenda)){

            $nodoAddenda = $root->addChild ( 'cfdi:Addenda', true);
            
            if(isset($dataFactura->addenda->inbursa) && $dataFactura->addenda->inbursa != null){

                $inbursa = $dataFactura->addenda->inbursa;
                
                $nodoReferenciaReceptor = $nodoAddenda->addChild('ReferenciaReceptor', true);
                
                $nodoReferenciaReceptor->addChild('Siniestro', true, [
                    'Emisor' => $inbursa->emisor,
                    'Numero' => $inbursa->numero,
                    'Afectado' => $inbursa->afectado
                ]);
                
                $nodoReferenciaReceptor->addChild('Deducible', true, [
                    'Importe' => Utils::formatNumber($inbursa->deducible, 2)
                ]);
                
                $nodoReferenciaReceptor->addChild('Descuento', true, [
                    'Importe' => Utils::formatNumber($inbursa->descuento, 2)
                ]);
                
                $nodoReferenciaReceptor->addChild('TotalManoObra', true, [
                    'Importe' => Utils::formatNumber($inbursa->manaoObra, 2)
                ]);
                
                $nodoReferenciaReceptor->addChild('TotalRefacciones', true, [
                    'Importe' => Utils::formatNumber($inbursa->refacciones, 2)
                ]);
                
                $nodoReferenciaReceptor->addChild('FechaEntregado', true, [
                    'Fecha' => date('d/m/Y', strtotime($inbursa->fechaEntrega))
                ]);

            }else{

                $nodoFondoGarantia = $nodoAddenda->addChild ( 'FondoGarantia:AddendaFondoGarantia', true, [
                    'xmlns:xsi' => 'http://www.w3.org/2001/XMLSchema',
                    "xsi:schemaLocation" => "http://201.148.157.232:81/Content/FondoGarantia http://201.148.157.232:81/Content/FondoGarantia/AddendaFondoGarantia.xsd",
                    'Version' => '1.0'
                ]);

                $nodoFondoGarantia->addChild ( 'FondoGarantia:FondoGarantia', true, [
                    'Importe' => $dataFactura->addenda->Importe,
                    "PeriodoEnMeses" => $dataFactura->addenda->PeriodoEnMeses,
                    'Descripcion' => $dataFactura->addenda->Descripcion
                ]);
            }

        }

		if(!isset($dataFactura->sello)){
			$sello = self::obtenerSello($xml, $proveedor->keypem);
			$rootNode->setAttribute ( 'Sello', $sello );
		}

		return $xml;
	}

    public static function createXML40($dataFactura, $proveedor) {


        $versionComercionXSD = '11';
        if(time() >
            strtotime('2024-03-08')){
            $versionComercionXSD = '20';
        }

        $cartaVersion = '20';
        $cartaVersionS = '2.0';
        if(strtotime($dataFactura->timeStamp) >
            strtotime('2024-07-17')){
            $cartaVersion = '31';
            $cartaVersionS = '3.1';
        }else if(time() >
			strtotime('2024-03-31')){
			$cartaVersion = '30';
			$cartaVersionS = '3.0';
		}


        $cfdiNS = new FluidNamespace ( 'cfdi', 'http://www.sat.gob.mx/cfd/4' );
        $cartaNS = new FluidNamespace ( "cartaporte$cartaVersion", "http://www.sat.gob.mx/CartaPorte$cartaVersion" );
        $xsiNS = new FluidNamespace ( 'xsi', 'http://www.w3.org/2001/XMLSchema-instance' );
        $xsiSLNS = new FluidNamespace ( 'schemaLocation', 'http://www.sat.gob.mx/cfd/4 http://www.sat.gob.mx/sitio_internet/cfd/4/cfdv40.xsd' );
        $tfdNs = new FluidNamespace ( 'tfd', 'http://www.sat.gob.mx/TimbreFiscalDigital' );
        $pagosNs = new FluidNamespace ( 'pago20', 'http://www.sat.gob.mx/Pagos20' );
        $impLNs = new FluidNamespace ( 'implocal', 'http://www.sat.gob.mx/implocal' );
        $nomLNs = new FluidNamespace ( 'nomina12', 'http://www.sat.gob.mx/nomina12' );
        $cceLNs = new FluidNamespace ( "cce$versionComercionXSD", "http://www.sat.gob.mx/ComercioExterior$versionComercionXSD" );
        $fondoNs = new FluidNamespace ( 'FondoGarantia', 'http://201.148.157.232:81/Content/FondoGarantia' );
        $ieduNs = new FluidNamespace ( 'iedu', 'http://www.sat.gob.mx/iedu' );
        $leyendasNs = new FluidNamespace ( 'leyendasFisc', 'http://www.sat.gob.mx/leyendasFiscales' );
        $ineNs = new FluidNamespace ( 'ine', 'http://www.sat.gob.mx/ine' );


        $xml = new FluidXml ( null ); // A document without a root node.
        $xml->namespace ( $xsiNS, $cfdiNS );
        $xml->namespace ( $tfdNs, $cfdiNS );
        $xml->namespace ( $pagosNs, $cfdiNS );
        if(isset($dataFactura->cartaPorte) && $dataFactura->cartaPorte != null){
            $xml->namespace ( $cartaNS, $cfdiNS );
        }

        if(isset($dataFactura->addenda) && $dataFactura->addenda != null){
            $xml->namespace ( $fondoNs, $cfdiNS );
        }

        if($dataFactura->tipoDeComprobante != 'P'){
            $xml->namespace ( $nomLNs, $cfdiNS );

            $xml->namespace ( $cceLNs, $cfdiNS );
        }

        if(!empty($dataFactura->norma)){
            $xml->namespace ( $leyendasNs, $cfdiNS );
        }

        if(isset($dataFactura->ine) && $dataFactura->ine != null){
            $xml->namespace ( $ineNs, $cfdiNS );
        }

        $listNameSpaces = "http://www.sat.gob.mx/cfd/4 http://www.sat.gob.mx/sitio_internet/cfd/4/cfdv40.xsd ";

        if((isset($dataFactura->impuestosTrasladadosLclTotal) && $dataFactura->impuestosTrasladadosLclTotal > 0)
            || isset($dataFactura->impuestosRetenidosLclTotal) && $dataFactura->impuestosRetenidosLclTotal > 0){
            $xml->namespace ( $impLNs, $cfdiNS );
            $listNameSpaces .= " http://www.sat.gob.mx/implocal http://www.sat.gob.mx/sitio_internet/cfd/implocal/implocal.xsd";
        }

        if($dataFactura->tipoDeComprobante == 'N'){
            $listNameSpaces .= " http://www.sat.gob.mx/nomina12 http://www.sat.gob.mx/sitio_internet/cfd/nomina/nomina12.xsd";
        }

        if($dataFactura->tipoDeComprobante == 'P'){
            $listNameSpaces .= " http://www.sat.gob.mx/Pagos20 http://www.sat.gob.mx/sitio_internet/cfd/Pagos/Pagos20.xsd";
        }

        if(isset($dataFactura->comercio) && $dataFactura->comercio != null){
            $listNameSpaces .= " http://www.sat.gob.mx/ComercioExterior$versionComercionXSD http://www.sat.gob.mx/sitio_internet/cfd/ComercioExterior$versionComercionXSD/ComercioExterior$versionComercionXSD.xsd";
        }

        if(isset($dataFactura->conceptos[0]->alumno) && $dataFactura->conceptos[0]->alumno != null){
            $xml->namespace ( $ieduNs );
            $listNameSpaces .= ' http://www.sat.gob.mx/iedu http://www.sat.gob.mx/sitio_internet/cfd/iedu/iedu.xsd';
        }

        if(isset($dataFactura->cartaPorte) && $dataFactura->cartaPorte != null){
            $listNameSpaces .= " http://www.sat.gob.mx/CartaPorte$cartaVersion http://www.sat.gob.mx/sitio_internet/cfd/CartaPorte/CartaPorte$cartaVersion.xsd";
        }

        if(!empty($dataFactura->norma)){
            $listNameSpaces .= " http://www.sat.gob.mx/leyendasFiscales http://www.sat.gob.mx/sitio_internet/cfd/leyendasFiscales/leyendasFisc.xsd";
        }

        if(isset($dataFactura->ine) && $dataFactura->ine != null){
            $listNameSpaces .= " http://www.sat.gob.mx/ine http://www.sat.gob.mx/sitio_internet/cfd/ine/ine11.xsd";
        }

        $root = $xml->addChild ( 'cfdi:Comprobante', true );

        $rootNode = $root [0];

        // Agrenado namespaces
        $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$cfdiNS->id()}", $cfdiNS->uri () );
        $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$xsiNS->id()}", $xsiNS->uri () );


        if($dataFactura->tipoDeComprobante == 'P'){
            $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$pagosNs->id()}", $pagosNs->uri () );
        }

        if((isset($dataFactura->impuestosTrasladadosLclTotal) && $dataFactura->impuestosTrasladadosLclTotal > 0)
            || isset($dataFactura->impuestosRetenidosLclTotal) && $dataFactura->impuestosRetenidosLclTotal > 0){
            $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$impLNs->id()}", $impLNs->uri () );
        }

        if($dataFactura->tipoDeComprobante == 'N'){
            $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$nomLNs->id()}", $nomLNs->uri () );
        }

        if(isset($dataFactura->comercio) && $dataFactura->comercio != null){
            $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$cceLNs->id()}", $cceLNs->uri () );
        }

        if(isset($dataFactura->cartaPorte) && $dataFactura->cartaPorte != null){
            $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$cartaNS->id()}", $cartaNS->uri () );
        }

        if(isset($dataFactura->conceptos[0]->alumno) && $dataFactura->conceptos[0]->alumno != null){
            $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$ieduNs->id()}", $ieduNs->uri () );
        }

        if(!empty($dataFactura->norma)){
            $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$leyendasNs->id()}", $leyendasNs->uri () );
        }

        if(isset($dataFactura->ine) && $dataFactura->ine != null){
            $rootNode->setAttributeNS ( 'http://www.w3.org/2000/xmlns/', "xmlns:{$ineNs->id()}", $ineNs->uri () );
        }

        $rootNode->setAttribute ( 'xsi:schemaLocation', $listNameSpaces );

        $fechaEmision = date_format ( new \DateTime ( $dataFactura->timeStamp ), 'Y-m-d\TH:i:s' );

        // Creando el nodo Comprobante
        $totalDocumento = 0;
        $subTotalDocumento = 0;
        if($dataFactura->tipoDeComprobante != 'P' && $dataFactura->tipoDeComprobante != 'T'){
            $totalDocumento = Utils::formatNumber ( $dataFactura->total, 2 );
            $subTotalDocumento = Utils::formatNumber ( $dataFactura->subtotal, 2 );
        }

        $moneda = $dataFactura->tipoDeComprobante == 'T' ? 'XXX' : $dataFactura->moneda;

        $rootNode->setAttribute ( 'Version', '4.0' );
        $rootNode->setAttribute ( 'Serie', $dataFactura->serie );
        $rootNode->setAttribute ( 'Folio', $dataFactura->folio );
        $rootNode->setAttribute ( 'Fecha', $fechaEmision );
        $rootNode->setAttribute ( 'NoCertificado', $dataFactura->noCertificado );
        $rootNode->setAttribute ( 'Certificado', $dataFactura->certificado );
        $rootNode->setAttribute ( 'SubTotal', $subTotalDocumento);
        $rootNode->setAttribute ( 'Moneda', $moneda );
        $rootNode->setAttribute ( 'Total',  $totalDocumento);
        $rootNode->setAttribute ( 'TipoDeComprobante', $dataFactura->tipoDeComprobante );
        $rootNode->setAttribute ( 'Exportacion', $dataFactura->exportacion );
        $rootNode->setAttribute ( 'LugarExpedicion', $dataFactura->lugarExpedicion );

        if($dataFactura->formaPago != '' && $dataFactura->tipoDeComprobante != 'T')
            $rootNode->setAttribute ( 'FormaPago', $dataFactura->formaPago );

        if($dataFactura->metodoDePago != '' && $dataFactura->tipoDeComprobante != 'T')
            $rootNode->setAttribute ( 'MetodoPago', $dataFactura->metodoDePago );

        if($dataFactura->condicionesDePago != "" && $dataFactura->tipoDeComprobante != 'T')
            $rootNode->setAttribute ( 'CondicionesDePago', $dataFactura->condicionesDePago );


        if ($dataFactura->descuento > 0) {
            $montoDescuento = Utils::formatNumber ( $dataFactura->descuento, $dataFactura->precission );
            $rootNode->setAttribute ( 'Descuento', $montoDescuento );
        }

        if($dataFactura->moneda != 'MXN' && $dataFactura->moneda != 'XXX')
            $rootNode->setAttribute ( 'TipoCambio', Utils::formatNumber ( $dataFactura->tipoCambio, $dataFactura->precission ) );



        if(count($dataFactura->facturasRelacionadas) > 0){

            if(!empty($dataFactura->tipoFacturasRelacionadas)) {
                // Viejo comportamiento
                $nodoRelacionados = $root->addChild('cfdi:CfdiRelacionados', true, [
                    'TipoRelacion' => $dataFactura->tipoFacturasRelacionadas
                ]);

                foreach ($dataFactura->facturasRelacionadas as $facturaRelacionada) {

                    $nodoRelacionados->addChild('cfdi:CfdiRelacionado', true, [
                        'UUID' => $facturaRelacionada->uuid
                    ]);

                }
            }else{

                //Nuevo comportamiento
                foreach ($dataFactura->facturasRelacionadas as $facturaRelacionada) {

                    $nodoRelacionados = $root->addChild('cfdi:CfdiRelacionados', true, [
                        'TipoRelacion' => $facturaRelacionada->tipoRelacion
                    ]);

                    $nodoRelacionados->addChild('cfdi:CfdiRelacionado', true, [
                        'UUID' => $facturaRelacionada->uuid
                    ]);

                }
            }
        }

        if(isset($dataFactura->infGlobal) && $dataFactura->infGlobal != null){

            $infoGlobal = array('Año' => $dataFactura->infGlobal->anio,
                'Meses' => $dataFactura->infGlobal->meses,
                'Periodicidad' => $dataFactura->infGlobal->periodicidad);

            $root->addChild ( 'cfdi:InformacionGlobal', true, $infoGlobal);
        }


        $emisorObj = [
            'Nombre' => $dataFactura->nmbEmisor,
            'Rfc' => $dataFactura->rFCEmisor,
            'RegimenFiscal' => $dataFactura->regimenFiscal];

        /*
        if($dataFactura->tipoPersona ==  2 && isset($dataFactura->curp) && $dataFactura->curp != ''){
            $emisorObj['Curp'] = $dataFactura->curp;
        }
        */

        $root->addChild ( 'cfdi:Emisor', true, $emisorObj);

        $recepArray = [
            'Nombre' => $dataFactura->nmbRecep,
            'Rfc' => $dataFactura->rFCRecep,
            'UsoCFDI' => $dataFactura->usoCFDI,
            'RegimenFiscalReceptor' => $dataFactura->regimenFiscalReceptor,
            'DomicilioFiscalReceptor' => $dataFactura->domicilioFiscalReceptor
        ];

        if($dataFactura->numRegIdTrib != '')
            $recepArray['NumRegIdTrib'] = $dataFactura->numRegIdTrib;

        if($dataFactura->pais != '')
            $recepArray['ResidenciaFiscal'] = $dataFactura->pais;

        $root->addChild ( 'cfdi:Receptor', true, $recepArray);

        $conceptosNode = $root->addChild ( 'cfdi:Conceptos', true );

        foreach ( $dataFactura->conceptos as $concepto ) {

            $agregarImpuestos = '';

            $importe = 0;
            $valUnit = 0;
            if($dataFactura->tipoDeComprobante != 'P'){
                $importe = Utils::formatNumber ( $concepto->prcNetoItem, $dataFactura->precission );
                $valUnit = Utils::formatNumber ( $concepto->montoNetoItem, $dataFactura->precission );
            }

            $attrConcepto = [
                'Cantidad' => $concepto->qtyItem,
                'ClaveProdServ' => $concepto->clave,
                'ClaveUnidad' => $concepto->unmdItem,
                'Descripcion' => $concepto->dscItem,
                'ObjetoImp' => $concepto->objetoImp,
                'Importe' => $importe,
                'ValorUnitario' => $valUnit
            ];

            if($concepto->cdgItem != '')
                $attrConcepto['NoIdentificacion'] = $concepto->cdgItem;

            if($concepto->descuento > 0)
                $attrConcepto['Descuento'] = Utils::formatNumber ($concepto->descuento, $dataFactura->precission);




            $nodoConcepto = $conceptosNode->addChild ( 'cfdi:Concepto', true, $attrConcepto);

            if(count($concepto->impuestosTrasladados) > 0 || count($concepto->impuestosRetenidos) > 0){

                $impuestosNode = $nodoConcepto->addChild ( 'cfdi:Impuestos', true );

                //Impuestos traslados
                if(count($concepto->impuestosTrasladados) > 0){

                    $trasladosNode = $impuestosNode->addChild ( 'cfdi:Traslados', true );

                    foreach($concepto->impuestosTrasladados as $impTraslado){

                        $atributos = [
                            'Base' => Utils::formatNumber ( $impTraslado->base, $dataFactura->precission ),
                            'Impuesto' => $impTraslado->tipoImp,
                            'TipoFactor' => $impTraslado->factor
                        ];

                        if($impTraslado->factor != 'Exento'){
                            $atributos['TasaOCuota'] = Utils::formatNumber ( $impTraslado->tasaImp, 6 );
                            $atributos['Importe'] = Utils::formatNumber ($impTraslado->montoImp,$dataFactura->precission);
                        }

                        $trasladosNode->addChild ( 'cfdi:Traslado', true, $atributos);

                    }
                }

                //Impuestos retenidos
                if(count($concepto->impuestosRetenidos) > 0){

                    $retencionesNode = $impuestosNode->addChild ( 'cfdi:Retenciones', true );

                    foreach($concepto->impuestosRetenidos as $impRetenido){

                        $attributos = [
                            'Base' => Utils::formatNumber ( $impRetenido->base, $dataFactura->precission ),
                            'Impuesto' => $impRetenido->tipoImp,
                            'TipoFactor' => $impRetenido->factor
                        ];

                        if($impRetenido->factor != 'Exento'){
                            $attributos['TasaOCuota'] = Utils::formatNumber ( $impRetenido->tasaImp, 6 );
                            $attributos['Importe'] = Utils::formatNumber ( $impRetenido->montoImp, $dataFactura->precission );
                        }

                        $retencionesNode->addChild ( 'cfdi:Retencion', true, $attributos);
                    }
                }

            }

            if(isset($concepto->alumno) && $concepto->alumno != null){

                $attributos = [
                    'CURP' => $concepto->alumno->curp,
                    'autRVOE' => $concepto->alumno->autRVOE,
                    'nivelEducativo' => $concepto->alumno->nivelEducativo,
                    'nombreAlumno' => $concepto->alumno->nombreAlumno,
                    'version' => '1.0',
                ];
                if($concepto->alumno->rfcPago != null && $concepto->alumno->rfcPago != ''){
                    $attributos['rfcPago'] = $concepto->alumno->rfcPago;
                }

                $nodoComplementConcepto = $nodoConcepto->addChild ( 'cfdi:ComplementoConcepto', true);
                $nodoComplementConcepto->addChild ( 'iedu:instEducativas', true, $attributos);

            }


            if(isset($dataFactura->cuentaPredial) && $dataFactura->cuentaPredial != ''){
                $nodoConcepto->addChild ( 'cfdi:CuentaPredial', true, ['Numero' => $dataFactura->cuentaPredial] );
            }

        }


        if(count($dataFactura->impuestosTrasladados) > 0 || count($dataFactura->impuestosRetenidos) > 0){

            $impuestosGeneralesNode = $root->addChild ( 'cfdi:Impuestos', true);

            if(count($dataFactura->impuestosRetenidos) > 0){

                $impuestosGeneralesNode->setAttribute ( 'TotalImpuestosRetenidos', Utils::formatNumber ( $dataFactura->totalImpRetenidos, 2 ) );

                $retencionesGralNode = $impuestosGeneralesNode->addChild ( 'cfdi:Retenciones', true );

                foreach ($dataFactura->impuestosRetenidos as $impRetencion){
                    $retencionesGralNode->addChild ( 'cfdi:Retencion', true, [
                        'Impuesto' => $impRetencion->impuesto,
                        'Importe' => Utils::formatNumber ( $impRetencion->importe, 2 )
                    ]);
                }

            }

            if(count($dataFactura->impuestosTrasladados) > 0){

                if(count($dataFactura->impuestosTrasladados) > 1 ||
                    (count($dataFactura->impuestosTrasladados) == 1 && $dataFactura->impuestosTrasladados[0]->tipoFactor != 'Exento')){
                    $impuestosGeneralesNode->setAttribute ( 'TotalImpuestosTrasladados', Utils::formatNumber ( $dataFactura->totalImpTrasladados, 2 ) );
                }

                $retencionesGralNode = $impuestosGeneralesNode->addChild ( 'cfdi:Traslados', true );

                foreach ($dataFactura->impuestosTrasladados as $impTraslado){
                    $arr = [
                        'Impuesto' => $impTraslado->impuesto,
                        'Base' => Utils::formatNumber($impTraslado->base, 2),
                        'TipoFactor' => $impTraslado->tipoFactor,
                    ];

                    if($impTraslado->tipoFactor != 'Exento'){
                        $arr['TasaOCuota'] = Utils::formatNumber($impTraslado->tasaOCuota, 6);
                        $arr['Importe'] = Utils::formatNumber($impTraslado->importe, 2);
                    }

                    $retencionesGralNode->addChild ( 'cfdi:Traslado', true, $arr);
                }
            }


        }

        if(isset($dataFactura->sello) ||
            (isset($dataFactura->pagos) && count($dataFactura->pagos) > 0) ||
            (count($dataFactura->impuestosRetenidosLcl) > 0 || count($dataFactura->impuestosTrasladadosLcl) > 0) || isset($dataFactura->nomina)
            || (isset($dataFactura->comercio) && $dataFactura->comercio != null)
            || (isset($dataFactura->cartaPorte) && $dataFactura->cartaPorte != null)
            || !empty($dataFactura->norma)
            || (isset($dataFactura->ine) && $dataFactura->ine != null)) {


            $nodoComplemento = $root->addChild ( 'cfdi:Complemento', true );


            if( (isset($dataFactura->impuestosRetenidosLcl) && count($dataFactura->impuestosRetenidosLcl)) > 0
                || (isset($dataFactura->impuestosTrasladadosLcl) && count($dataFactura->impuestosTrasladadosLcl) > 0) ){

                $properties = array('version' => '1.0');
                $properties['TotaldeRetenciones'] = Utils::formatNumber ( $dataFactura->impuestosRetenidosLclTotal, $dataFactura->precission );
                $properties['TotaldeTraslados'] = Utils::formatNumber ( $dataFactura->impuestosTrasladadosLclTotal, $dataFactura->precission );

                $nodoImpuestosLocales = $nodoComplemento->addChild('implocal:ImpuestosLocales', true, $properties);

                if(isset($dataFactura->impuestosRetenidosLcl) && count($dataFactura->impuestosRetenidosLcl) > 0){
                    foreach ($dataFactura->impuestosRetenidosLcl as $impuestoRetLocal){
                        $nodoImpuestosLocales->addChild('implocal:RetencionesLocales', '',
                            ['ImpLocRetenido' => $impuestoRetLocal->tipoImp,
                                'Importe' => Utils::formatNumber($impuestoRetLocal->montoImp, 2),
                                'TasadeRetencion' => Utils::formatNumber($impuestoRetLocal->tasaImp, $dataFactura->precission)]);
                    }
                }

                if(isset($dataFactura->impuestosTrasladadosLcl) && count($dataFactura->impuestosTrasladadosLcl) > 0){
                    foreach ($dataFactura->impuestosTrasladadosLcl as $impuestoTrasLocal){
                        $nodoImpuestosLocales->addChild('implocal:TrasladosLocales', '', ['ImpLocTrasladado' => $impuestoTrasLocal->tipoImp, 'Importe' => Utils::formatNumber($impuestoTrasLocal->montoImp, 2), 'TasadeTraslado' => Utils::formatNumber($impuestoTrasLocal->tasaImp, $dataFactura->precission)]);
                    }
                }

            }

            if(!empty($dataFactura->norma)){
                $nodoLeyendasFiscales = $nodoComplemento->addChild('leyendasFisc:LeyendasFiscales', true, array('version' => '1.0'));
                $properties = array('norma' => $dataFactura->norma, 'textoLeyenda' => $dataFactura->leyenda);
                $nodoLeyendasFiscales->addChild('leyendasFisc:Leyenda', true, $properties);
            }


            if(isset($dataFactura->ine) && $dataFactura->ine != null){

                $arrayINE = array('TipoProceso' => $dataFactura->ine->tipoProceso, 'Version' => "1.1");
                if($dataFactura->ine->tipoProceso == 'Ordinario'){
                    $arrayINE['TipoComite'] = $dataFactura->ine->comite;
                }

                $nodoINE = $nodoComplemento->addChild('ine:INE', true, $arrayINE);


                //if($dataFactura->ine->comite != 'Ejecutivo Nacional'){
                    $arrayEntidad = array('ClaveEntidad' => $dataFactura->ine->claveEntidad);

                    //if($dataFactura->ine->comite != 'Ejecutivo Estatal'){
                        $arrayEntidad['Ambito'] = $dataFactura->ine->ambito;
                    //}

                    $nodoEntidad = $nodoINE->addChild('ine:Entidad', true, $arrayEntidad);
                    if($dataFactura->ine->idContabilidad > 0){
                        $nodoEntidad->addChild('ine:Contabilidad', true, array('IdContabilidad' => $dataFactura->ine->idContabilidad));
                    //}
                }


            }

            if(isset($dataFactura->cartaPorte) && $dataFactura->cartaPorte != null){

                $cartaPorte = $dataFactura->cartaPorte;

                $arrayCartaPorte = array('TranspInternac' => $cartaPorte->transpInternac, 'Version' => $cartaVersionS);

                if($cartaVersion == '30' || $cartaVersion == '31'){
                    $arrayCartaPorte['IdCCP'] = $cartaPorte->idccp;
                }

                if(isset($cartaPorte->totalDistanciaRecorrida) && $cartaPorte->totalDistanciaRecorrida > 0)
                    $arrayCartaPorte['TotalDistRec'] = Utils::formatNumber($cartaPorte->totalDistanciaRecorrida, 2);

                if($cartaPorte->transpInternac == 'Si' || $cartaPorte->transpInternac == 'Sí'){
                    $arrayCartaPorte['ViaEntradaSalida'] = $cartaPorte->viaEntradaSalida;
                    $arrayCartaPorte['EntradaSalidaMerc'] = $cartaPorte->entradaSalidaMerc;
                    $arrayCartaPorte['PaisOrigenDestino'] = $cartaPorte->paisOrigenDestino;
                }

                $nodoCartaPorte = $nodoComplemento->addChild("cartaporte$cartaVersion:CartaPorte", true, $arrayCartaPorte);

				if($cartaPorte->transpInternac == 'Sí'){
					$nodoRegimenesAduaneros = $nodoCartaPorte->addChild("cartaporte$cartaVersion:RegimenesAduaneros", true);
					$nodoRegimenesAduaneros->addChild("cartaporte$cartaVersion:RegimenAduaneroCCP", true, array('RegimenAduanero' => 'EXD'));
				}

                if($cartaPorte->ubicaciones != null && count($cartaPorte->ubicaciones) > 0){

                    $nodoUbicaciones = $nodoCartaPorte->addChild ( "cartaporte$cartaVersion:Ubicaciones", true );

                    foreach ($cartaPorte->ubicaciones as $ubicacion){

                        $ubicacionArray = array('FechaHoraSalidaLlegada' => $ubicacion->fechaHoraSalidaLlegada, 'RFCRemitenteDestinatario' => $ubicacion->rFCRemitenteDestinatario, 'TipoUbicacion' => $ubicacion->tipoUbicacion);

                        if(!empty($ubicacion->totalDistRec))
                            $ubicacionArray['DistanciaRecorrida'] = Utils::formatNumber($ubicacion->totalDistRec);

						if(!empty($ubicacion->idUbicacion))
							$ubicacionArray['IDUbicacion'] = $ubicacion->idUbicacion;

						if(!empty($ubicacion->nombreDestinatario))
							$ubicacionArray['NombreRemitenteDestinatario'] = $ubicacion->nombreDestinatario;

                        if(!empty($ubicacion->numRegIdTrib)) {
                            $countriesISO = array('USA' => 'US');
                            $ubicacionArray['NumRegIdTrib'] = trim($ubicacion->numRegIdTrib);
                            $ubicacionArray['ResidenciaFiscal'] = $ubicacion->pais;
                        }


                        $nodoUbicacion = $nodoUbicaciones->addChild ( "cartaporte$cartaVersion:Ubicacion", true, $ubicacionArray);

                        if(!empty($ubicacion->pais)){
                            $domicilioU = array('Pais' => $ubicacion->pais, 'Estado' => $ubicacion->estado, 'CodigoPostal' => $ubicacion->codigoPostal);

                            if(!empty($ubicacion->calle)){
                                $domicilioU['Calle'] = $ubicacion->calle;
                            }
                            if(!empty($ubicacion->numeroExterior)){
                                $domicilioU['NumeroExterior'] = $ubicacion->numeroExterior;
                            }
                            if(!empty($ubicacion->numeroInterior)){
                                $domicilioU['NumeroInterior'] = $ubicacion->numeroInterior;
                            }
                            if(!empty($ubicacion->colonia)){
                                $domicilioU['Colonia'] = $ubicacion->colonia;
                            }
                            if(!empty($ubicacion->municipio)){
                                $domicilioU['Municipio'] = $ubicacion->municipio;
                            }

                            $nodoUbicacion->addChild ( "cartaporte$cartaVersion:Domicilio", true, $domicilioU);

                        }
                    }
                }

                $mercanciasArray = array('NumTotalMercancias' => count($cartaPorte->mercancias), 'PesoBrutoTotal' => $cartaPorte->pesoBrutoTotal, 'UnidadPeso' => $cartaPorte->unidadPeso);
                $nodoMercancias = $nodoCartaPorte->addChild("cartaporte$cartaVersion:Mercancias", true, $mercanciasArray);

                $arrayWithDanger = array(11162100,11162101,11162102,11162104,11162105,11162107,11162108,11162109,11162110,11162111,11162112,11162113,11162114,11162115,11162116,11162117,11162118,11162119,11162120,11162121,11162122,11162123,11162124,11162125,11162126,11162127,10171607,10171601,10171504,10171611,24112902,10171600);

                foreach ( $cartaPorte->mercancias as $concepto ) {

                    $importe = 0;
                    if($concepto->valorMercancia > 0){
                        $importe = Utils::formatNumber ( $concepto->valorMercancia, $dataFactura->precission );
                    }

                    $attrConcepto = [
                        'Cantidad' => $concepto->cantidad,
                        'ClaveUnidad' => $concepto->claveUnidad,
                        'BienesTransp' => $concepto->bienesTransp,
                        'Descripcion' => $concepto->descripcion,
                        'PesoEnKg' => $concepto->peso
                    ];

                    if(isset($concepto->fraccionArancelaria) && !empty($concepto->fraccionArancelaria)){
                        $attrConcepto['FraccionArancelaria'] = $concepto->fraccionArancelaria;
                    }

					if($cartaPorte->transpInternac == 'Sí'){
						$attrConcepto['TipoMateria'] = '01';
					}

                    if(in_array($concepto->bienesTransp, $arrayWithDanger)){
                        $attrConcepto['MaterialPeligroso'] = 'No';
                    }

                    if($importe > 0){
                        $attrConcepto['ValorMercancia'] = $importe;
                        $attrConcepto['Moneda'] = $dataFactura->moneda;
                    }


                    $nodoMercancia = $nodoMercancias->addChild("cartaporte$cartaVersion:Mercancia", true, $attrConcepto);

                    if(isset($concepto->pedimento) &&  !empty($concepto->pedimento)){
                        $nodoMercancia->addChild("cartaporte$cartaVersion:Pedimentos", true, array('Pedimento' => $concepto->pedimento));
                    }
                }

                if($cartaPorte->autotransporte != null){

                    $autotransporte = $cartaPorte->autotransporte;

                    $array = array('PermSCT' => $autotransporte->tipoPermiso);
                    if(!empty($autotransporte->noPermiso))
                        $array['NumPermisoSCT'] = $autotransporte->noPermiso;

                    $nodoAutotransporte = $nodoMercancias->addChild("cartaporte$cartaVersion:Autotransporte", true,
                        $array);

                    $indentVehiArray = array('AnioModeloVM' => $autotransporte->anioModeloVM, 'ConfigVehicular' => $autotransporte->configTransporte, 'PlacaVM' => $autotransporte->placaVM);
                    if($cartaVersion == '30' || $cartaVersion == '31'){

						$persoVehicular = $autotransporte->pesoBrutoVehicular;
						if (floor($persoVehicular) == $persoVehicular) {
							$persoVehicular = number_format($persoVehicular, 1, '.', '');
						}

                        $indentVehiArray['PesoBrutoVehicular'] = $persoVehicular;
                    }
                    $nodoAutotransporte->addChild("cartaporte$cartaVersion:IdentificacionVehicular", true, $indentVehiArray);

                    $nodoAutotransporte->addChild("cartaporte$cartaVersion:Seguros", true,
                        array('AseguraRespCivil' => $autotransporte->aseguraRespCivil, 'PolizaRespCivil' => $autotransporte->polizaRespCivil));

                    if($autotransporte->remolques != null && count($autotransporte->remolques) > 0){
                        $nodoRemolques = $nodoAutotransporte->addChild("cartaporte$cartaVersion:Remolques", true);
                        foreach ($autotransporte->remolques as $remolque){
                            $nodoRemolques->addChild("cartaporte$cartaVersion:Remolque", true,
                                array('Placa' => $remolque->placa, 'SubTipoRem' => $remolque->subTipoRem));
                        }
                    }
                }

                if($cartaPorte->figurasTransporte != null){

                    $nodoFigurasTransporte = $nodoCartaPorte->addChild ( "cartaporte$cartaVersion:FiguraTransporte", true);

                    foreach ($cartaPorte->figurasTransporte as $figuraTransporte){

                        $figuraArray = array('TipoFigura' => $figuraTransporte->tipoFigura);

                        if(!empty($figuraTransporte->rFCFigura))
                            $figuraArray['RFCFigura'] = $figuraTransporte->rFCFigura;

                        if(!empty($figuraTransporte->numLicencia))
                            $figuraArray['NumLicencia'] = $figuraTransporte->numLicencia;

                        if(!empty($figuraTransporte->numRegIdTribFigura))
                            $figuraArray['NumRegIdTribFigura'] = $figuraTransporte->numRegIdTribFigura;

                        if(!empty($figuraTransporte->nombreFigura))
                            $figuraArray['NombreFigura'] = $figuraTransporte->nombreFigura;

                        $notoTipoFigura = $nodoFigurasTransporte->addChild ( "cartaporte$cartaVersion:TiposFigura", true, $figuraArray);

                        if($figuraTransporte->tipoFigura == '03'|| $figuraTransporte->tipoFigura == '02' ){
                            $notoTipoFigura->addChild ( "cartaporte$cartaVersion:PartesTransporte", true, array('ParteTransporte' => $figuraTransporte->parteTransporte));

                            $notoTipoFigura->addChild ( "cartaporte$cartaVersion:Domicilio", true,
                                 array('CodigoPostal' => $figuraTransporte->codigoPostal, 'Estado' => $figuraTransporte->estado, 'Pais' => $figuraTransporte->pais));




                        }

                    }
                }
            }


            if(isset($dataFactura->pagos) && count($dataFactura->pagos) > 0){

                $nodoPagos = $nodoComplemento->addChild ( 'pago20:Pagos', true, [
                    'Version' => '2.0'
                ]);

                $attrTotales = array('MontoTotalPagos' => Utils::formatNumber ($dataFactura->totalesPagos->montoTotalPagos));


                if($dataFactura->totalesPagos->totalRetencionesIVA > 0 || (isset($dataFactura->totalesPagos->tieneRetencionIVA) && $dataFactura->totalesPagos->tieneRetencionIVA))
                    $attrTotales['TotalRetencionesIVA'] = Utils::formatNumber ($dataFactura->totalesPagos->totalRetencionesIVA);

                if($dataFactura->totalesPagos->totalRetencionesISR > 0)
                    $attrTotales['TotalRetencionesISR'] = Utils::formatNumber ($dataFactura->totalesPagos->totalRetencionesISR);

                if($dataFactura->totalesPagos->totalRetencionesIEPS > 0)
                    $attrTotales['TotalRetencionesIEPS'] = Utils::formatNumber ($dataFactura->totalesPagos->totalRetencionesIEPS);

                if($dataFactura->totalesPagos->totalTrasladosBaseIVA16 > 0){
                    $attrTotales['TotalTrasladosBaseIVA16'] = Utils::formatNumber ($dataFactura->totalesPagos->totalTrasladosBaseIVA16);
                    $attrTotales['TotalTrasladosImpuestoIVA16'] = Utils::formatNumber ($dataFactura->totalesPagos->totalTrasladosImpuestoIVA16);
                }
                if($dataFactura->totalesPagos->totalTrasladosBaseIVA8 > 0){
                    $attrTotales['TotalTrasladosBaseIVA8'] = Utils::formatNumber ($dataFactura->totalesPagos->totalTrasladosBaseIVA8);
                    $attrTotales['TotalTrasladosImpuestoIVA8'] = Utils::formatNumber ($dataFactura->totalesPagos->totalTrasladosImpuestoIVA8);
                }
                if($dataFactura->totalesPagos->totalTrasladosBaseIVA0 > 0){
                    $attrTotales['TotalTrasladosBaseIVA0'] = Utils::formatNumber ($dataFactura->totalesPagos->totalTrasladosBaseIVA0);
                    $attrTotales['TotalTrasladosImpuestoIVA0'] = Utils::formatNumber ($dataFactura->totalesPagos->totalTrasladosImpuestoIVA0);
                }
                if($dataFactura->totalesPagos->totalTrasladosBaseIVAExento > 0){
                    $attrTotales['TotalTrasladosBaseIVAExento'] = Utils::formatNumber ($dataFactura->totalesPagos->totalTrasladosBaseIVAExento);
                }




                $nodoPagos->addChild ( 'pago20:Totales', true, $attrTotales);


                foreach ($dataFactura->pagos as $pago){

                    $fechaPago = new \DateTime ( $pago->FechaPago );
                    $fechaPago = date_format ( $fechaPago, 'Y-m-d\TH:i:s' );

                    $pagoElment = [
                        'FechaPago' => $fechaPago,
                        'FormaDePagoP' => $pago->FormaDePagoP,
                        'MonedaP' => $pago->MonedaP,
                        'Monto' => Utils::formatNumber ( $pago->Monto, 2 )
                    ];

                    if($pago->banco != ''){
                        $pagoElment['RfcEmisorCtaOrd'] = $pago->banco;
                    }

                    if($pago->cuenta != ''){
                        $pagoElment['CtaOrdenante'] = $pago->cuenta;
                    }

                    if(isset($pago->rfcEmisorCtaBen) && $pago->rfcEmisorCtaBen != null && $pago->rfcEmisorCtaBen != ''){
                        $pagoElment['RfcEmisorCtaBen'] = $pago->rfcEmisorCtaBen;
                    }

                    if(isset($pago->ctaBeneficiario) && $pago->ctaBeneficiario != null && $pago->ctaBeneficiario != ''){
                        $pagoElment['CtaBeneficiario'] = $pago->ctaBeneficiario;
                    }

                    if(isset($pago->noOperacion) && $pago->noOperacion != ''){
                        $pagoElment['NumOperacion'] = $pago->noOperacion;
                    }

                    if($pago->TipoCambioP > 1){
                        $pagoElment['TipoCambioP'] = $pago->TipoCambioP;
                    }else{
                        $pagoElment['TipoCambioP'] = 1;
                    }

                    /*
                    echo "Sergio flores";
                    print_r($pago);
                    exit;
                    */

                    $nodoPago = $nodoPagos->addChild ( 'pago20:Pago', true, $pagoElment);

                    if(count($pago->documentosRelacionado) > 0)
                        foreach ($pago->documentosRelacionado as $doctoRelacionado){

                            $equivalenciaDR = 1;
                            if($doctoRelacionado->MonedaDR == 'USD' && $pago->MonedaP == 'MXN' ){
                                $equivalenciaDR = Utils::formatNumber ($doctoRelacionado->ImpPagado / $doctoRelacionado->MontoPagoMoneda, 6);

                                //echo "1: " . ($doctoRelacionado->ImpPagado / $equivalenciaDR) . " -- 2: " . $doctoRelacionado->MontoPagoMoneda;

                                while($doctoRelacionado->ImpPagado / $equivalenciaDR > $doctoRelacionado->MontoPagoMoneda) {
                                        $equivalenciaDR += 0.000001;
                                }
                            }

                            $docRelData = [
                                'IdDocumento' => $doctoRelacionado->IdDocumento,
                                'MonedaDR' => $doctoRelacionado->MonedaDR,
                                'NumParcialidad' => $doctoRelacionado->NumParcialidad,
                                'ImpSaldoAnt' => Utils::formatNumber ( $doctoRelacionado->ImpSaldoAnt, 2 ),
                                'ImpPagado' => Utils::formatNumber ( $doctoRelacionado->ImpPagado, 2 ),
                                'ImpSaldoInsoluto' => Utils::formatNumber ( $doctoRelacionado->ImpSaldoInsoluto, 2 ),
                                'EquivalenciaDR' => $equivalenciaDR,
                                'ObjetoImpDR' => $doctoRelacionado->ObjetoImpDR
                            ];

                            /*
                            if($doctoRelacionado->TipoCambioDR != null)
                                $docRelData['TipoCambioDR'] = Utils::formatNumber ( $doctoRelacionado->TipoCambioDR, 2 );
                            */

                            $nodoDocumentoRelacionado = $nodoPago->addChild ( 'pago20:DoctoRelacionado', true, $docRelData);

                            if(!empty($doctoRelacionado->impuestos) &&
                                (count($doctoRelacionado->impuestos->impuestosRetenidos) > 0
                                || count($doctoRelacionado->impuestos->impuestosTrasladados) > 0)){

                                $nodoImpuestosDR = $nodoDocumentoRelacionado->addChild ( 'pago20:ImpuestosDR', true);


                                if(count($doctoRelacionado->impuestos->impuestosRetenidos) > 0 ){
                                    $nodoRetencionesDR = $nodoImpuestosDR->addChild ( 'pago20:RetencionesDR', true);

                                    foreach ($doctoRelacionado->impuestos->impuestosRetenidos as $impuestoRetenido){

                                        $impuestoRetenidoData = array(
                                            'BaseDR' => Utils::formatNumber ($impuestoRetenido->base),
                                            'ImpuestoDR' => $impuestoRetenido->impuesto,
                                            'TipoFactorDR' => $impuestoRetenido->tipoFactor,
                                        );

                                        if($impuestoRetenido->tipoFactor != 'Exento'){
                                            $impuestoRetenidoData['ImporteDR'] = Utils::formatNumber ($impuestoRetenido->importe);
                                            $impuestoRetenidoData['TasaOCuotaDR'] = Utils::formatNumber ( $impuestoRetenido->tasaCuota, 6);

                                        }

                                        $nodoRetencionesDR->addChild ( 'pago20:RetencionDR', true, $impuestoRetenidoData);

                                    }
                                }


                                if(count($doctoRelacionado->impuestos->impuestosTrasladados) > 0 ){
                                    $nodoTrasladosDR = $nodoImpuestosDR->addChild ( 'pago20:TrasladosDR', true);

                                    foreach ($doctoRelacionado->impuestos->impuestosTrasladados as $impuestoTrasladado){

                                        $impuestoTrasladadoData = array(
                                            'BaseDR' => Utils::formatNumber ($impuestoTrasladado->base),
                                            'ImpuestoDR' => $impuestoTrasladado->impuesto,
                                            'TipoFactorDR' => $impuestoTrasladado->tipoFactor,
                                        );

                                        if($impuestoTrasladado->tipoFactor != 'Exento'){
                                            $impuestoTrasladadoData['ImporteDR'] = Utils::formatNumber ($impuestoTrasladado->importe);
                                            $impuestoTrasladadoData['TasaOCuotaDR'] = Utils::formatNumber ( $impuestoTrasladado->tasaCuota, 6);

                                        }

                                        $nodoTrasladosDR->addChild ( 'pago20:TrasladoDR', true, $impuestoTrasladadoData);

                                    }
                                }

                            }

                        }

                    if( isset($pago->impuestos)
                        && (count($pago->impuestos->impuestosRetenidos) > 0
                        || count($pago->impuestos->impuestosTrasladados) > 0)){

                        $nodoImpuestosP = $nodoPago->addChild ( 'pago20:ImpuestosP', true);

                        if( count($pago->impuestos->impuestosRetenidos) > 0){
                            $nodoRetencionesP = $nodoImpuestosP->addChild ( 'pago20:RetencionesP', true);

                            foreach ($pago->impuestos->impuestosRetenidos as $retencion){
                                $retencionData = array(
                                    'ImporteP' => Utils::formatNumber ($retencion->importe),
                                    'ImpuestoP' => $retencion->impuesto);

                                $nodoRetencionesP->addChild ( 'pago20:RetencionP', true, $retencionData);
                            }
                        }

                        if( count($pago->impuestos->impuestosTrasladados) > 0){
                            $nodoRetencionesP = $nodoImpuestosP->addChild ( 'pago20:TrasladosP', true);

                            foreach ($pago->impuestos->impuestosTrasladados as $trasladado){

                                $impuestoTrasladadoData = array(
                                    'BaseP' => Utils::formatNumber ($trasladado->base, 2),
                                    'ImpuestoP' => $trasladado->impuesto,
                                    'TipoFactorP' => $trasladado->tipoFactor,
                                );

                                if($trasladado->tipoFactor != 'Exento'){
                                    $impuestoTrasladadoData['ImporteP'] = Utils::formatNumber ($trasladado->importe);
                                    $impuestoTrasladadoData['TasaOCuotaP'] = Utils::formatNumber ( $trasladado->tasaCuota, 6);

                                }

                                $nodoRetencionesP->addChild ( 'pago20:TrasladoP', true, $impuestoTrasladadoData);
                            }
                        }

                    }
                }
            }


            if(isset($dataFactura->nomina)){

                $nomina = $dataFactura->nomina;

                $receptor = $nomina['Receptor'];

                $arrayNomina = [
                    'Version' => $nomina['Version'],
                    'FechaPago' => $nomina['FechaPago'],
                    'FechaInicialPago' => $nomina['FechaInicialPago'],
                    'FechaFinalPago' => $nomina['FechaFinalPago'],
                    'NumDiasPagados' => $nomina['NumDiasPagados'],
                    'TipoNomina' => $nomina['TipoNomina']
                ];


                $OtrosPagos = $nomina['OtrosPagos'];

                if($nomina['TotalDeducciones'] > 0)
                    $arrayNomina['TotalDeducciones'] = $nomina['TotalDeducciones'];

                if($nomina['TotalPercepciones'] > 0)
                    $arrayNomina['TotalPercepciones'] = $nomina['TotalPercepciones'];

                if(count($OtrosPagos['listadoOtrosPagos']) > 0)
                    $arrayNomina['TotalOtrosPagos'] = $nomina['TotalOtrosPagos'];


                $nodoNomina = $nodoComplemento->addChild ( 'nomina12:Nomina', true, $arrayNomina);

                $emisorNomina = array();

                if(isset($receptor ['TipoContrato']) && $receptor ['TipoContrato'] != '09' && $receptor ['TipoContrato'] != '10' && $receptor ['TipoContrato'] != '99'){
                    $emisorNomina = ['RegistroPatronal' => $nomina['Emisor']['RegistroPatronal']];
                }

                if($dataFactura->tipoPersona ==  2 && isset($dataFactura->curp) && $dataFactura->curp != ''){
                    $emisorNomina['Curp'] = $dataFactura->curp;
                }

                $nodoNomina->addChild ( 'nomina12:Emisor', true, $emisorNomina);

                $arrayData = [];

                if (! empty ( $receptor ['Curp'] ))
                    $arrayData ['Curp'] = $receptor ['Curp'];

                if (! empty ( $receptor ['TipoContrato'] ))
                    $arrayData ['TipoContrato'] = trim ( $receptor ['TipoContrato'] );

                if (! empty ( $receptor ['TipoRegimen'] )) {
                    $arrayData ['TipoRegimen'] = $receptor ['TipoRegimen'];
                }

                if (! empty ( $receptor ['NumEmpleado'] ))
                    $arrayData ['NumEmpleado'] = $receptor ['NumEmpleado'];

                if (! empty ( $receptor ['PeriodicidadPago'] ))
                    $arrayData ['PeriodicidadPago'] = $receptor ['PeriodicidadPago'];

                if (! empty ( $receptor ['ClaveEntFed'] ))
                    $arrayData ['ClaveEntFed'] = trim ( $receptor ['ClaveEntFed'] );

                if (! empty ( $receptor ['NumSeguridadSocial'] ) && $receptor ['TipoRegimen'] != 10)
                    $arrayData ['NumSeguridadSocial'] = trim ( $receptor ['NumSeguridadSocial'] );

                if (! empty ( $receptor ['FechaInicioRelLaboral'] ) && $receptor ['TipoRegimen'] != 10)
                    $arrayData ['FechaInicioRelLaboral'] = trim ( $receptor ['FechaInicioRelLaboral'] );

                if (! empty ($receptor ['Antigüedad'] ) && $receptor ['TipoRegimen'] != 10)
                    $arrayData ['Antigüedad'] = $receptor['Antigüedad'];

                if (! empty ($receptor ['Puesto'] ))
                    $arrayData ['Puesto'] = trim ( $receptor ['Puesto'] );

                if (! empty ( $receptor ['SalarioBaseCotApor'] ) && $receptor ['SalarioBaseCotApor'] > 0)
                    $arrayData ['SalarioBaseCotApor'] = $receptor ['SalarioBaseCotApor'];

                if (! empty ( $receptor ['SalarioDiarioIntegrado'] ) && $receptor ['SalarioDiarioIntegrado'] > 0 && $receptor ['TipoRegimen'] != 10)
                    $arrayData ['SalarioDiarioIntegrado'] = $receptor ['SalarioDiarioIntegrado'];

                if (! empty ( $receptor ['RiesgoPuesto'] ) && $receptor ['TipoRegimen'] != 10) {
                    $arrayData ['RiesgoPuesto'] = $receptor ['RiesgoPuesto'];
                }


                $nodoNomina->addChild ( 'nomina12:Receptor', true, $arrayData);


                $percepciones = $nomina['Percepciones'];
                $deducciones = $nomina['Deducciones'];
                $OtrosPagos = $nomina['OtrosPagos'];
                $Incapacidades = $nomina['Incapacidades'];


                if(count($percepciones['listadoPercepciones']) > 0){

                    $totalesPercepciones = [];

                    $totalesPercepciones['TotalGravado'] = Utils::formatNumber($percepciones['totalPercepcionGravado']);

                    $totalesPercepciones['TotalExento'] = Utils::formatNumber($percepciones['totalPercepcionExcento']);

                    if($percepciones['totalSueldos'] > 0){
                        $totalesPercepciones['TotalSueldos'] = Utils::formatNumber($percepciones['totalSueldos']);
                    }

                    if($percepciones['totalSeparacionIndemnizacion'] > 0){
                        $totalesPercepciones['TotalSeparacionIndemnizacion'] = Utils::formatNumber($percepciones['totalSeparacionIndemnizacion']);
                    }

                    if($percepciones['totalJubilacionPension'] > 0){
                        $totalesPercepciones['TotalJubilacionPensionRetiro'] = Utils::formatNumber($percepciones['totalJubilacionPension']);
                    }


                    $nodoPercepciones = $nodoNomina->addChild ( 'nomina12:Percepciones', true, $totalesPercepciones);

                    foreach ($percepciones['listadoPercepciones'] as $percepcionObj){


                        $percepcionElement = array('TipoPercepcion' => $percepcionObj['TipoPercepcion'], 'Clave' => $percepcionObj['Clave'],
                            'Concepto' => $percepcionObj['Concepto'], 'ImporteGravado' => $percepcionObj['ImporteGravado'],
                            'ImporteExento' => $percepcionObj['ImporteExento']);


                        $nodoPercepcion = $nodoPercepciones->addChild ( 'nomina12:Percepcion', true, $percepcionElement);

                        if($percepcionObj['TipoPercepcion'] == '019'){
                            $horaExtra = array('Dias' => $percepcionObj['Dias'], 'TipoHoras' => str_pad ( $percepcionObj['TipoHoras'], 2, "0", STR_PAD_LEFT ),
                                'HorasExtra' => $percepcionObj['HorasExtra'], 'ImportePagado'  => $percepcionObj['ImportePagado']);

                            $nodoPercepcion->addChild ( 'nomina12:HorasExtra', true, $horaExtra);
                        }

                    }

                }


                if(count($deducciones['listadoDeducciones']) > 0){

                    $totalesDeducciones = [];

                    if($deducciones['totalOtrasDeducciones'] > 0){
                        $totalesDeducciones['TotalOtrasDeducciones'] = Utils::formatNumber($deducciones['totalOtrasDeducciones']);
                    }

                    if($deducciones['totalImpuestosRetenidos'] > 0){
                        $totalesDeducciones['TotalImpuestosRetenidos'] = Utils::formatNumber($deducciones['totalImpuestosRetenidos']);
                    }

                    $nodoDeducciones = $nodoNomina->addChild ( 'nomina12:Deducciones', true, $totalesDeducciones);

                    foreach ($deducciones['listadoDeducciones'] as $deduccionObj){
                        $nodoDeducciones->addChild ( 'nomina12:Deduccion', true, $deduccionObj);
                    }

                }


                if(count($OtrosPagos['listadoOtrosPagos']) > 0){

                    $nodoDeducciones = $nodoNomina->addChild ( 'nomina12:OtrosPagos', true);

                    foreach ($OtrosPagos['listadoOtrosPagos'] as $otroPago){

                        $nodoOtroPago = $nodoDeducciones->addChild ( 'nomina12:OtroPago', true, $otroPago);

                        if($otroPago['TipoOtroPago'] == '002'){
                            $nodoOtroPago->addChild ( 'nomina12:SubsidioAlEmpleo', true, array('SubsidioCausado' => $otroPago['Importe']));
                        }

                    }

                }

                if(count($Incapacidades['listadoIncapacidades']) > 0){

                    $nodoDeducciones = $nodoNomina->addChild ( 'nomina12:Incapacidades', true);

                    foreach ($Incapacidades['listadoIncapacidades'] as $incapacidad){
                        $nodoDeducciones->addChild ( 'nomina12:Incapacidad', true, $incapacidad);
                    }

                }


            }

            //Complemento comercio exterior
            if((isset($dataFactura->comercio) && $dataFactura->comercio != null)){

                $comercio = $dataFactura->comercio;

                $versionComercion = '1.1';
                if(time() >
                    strtotime('2024-03-08')){
                    $versionComercion = '2.0';
                }
                $properties = array('Version' => $versionComercion);
                $properties['ClaveDePedimento'] = $comercio->datos->clavePedimento;
                $properties['Incoterm'] = $comercio->datos->Incoterm;

				if($dataFactura->rFCEmisor == 'SME200803NS3' && $dataFactura->moneda == 'EUR'){
					$properties['NumeroExportadorConfiable'] = 'MX/0058-22';
				}

                if($versionComercion == '1.1'){
                    $properties['TipoOperacion'] = $comercio->datos->tipoOperacion;
                    $properties['Subdivision'] = 0;
                }

                $properties['CertificadoOrigen'] = 0;

                if(!empty($comercio->datos->motivoTraslado))
                    $properties['MotivoTraslado'] = $comercio->datos->motivoTraslado;


                if($dataFactura->moneda == 'USD' || $dataFactura->moneda == 'EUR'){
                    $properties['TipoCambioUSD'] = $dataFactura->tipoCambio;
                    $properties['TotalUSD'] = Utils::formatNumber ( $comercio->datos->totalUSD, $dataFactura->precission );
                }

                $nodoComercioE = $nodoComplemento->addChild("cce$versionComercionXSD:ComercioExterior", true, $properties);

                $datosEmisorCE = array();
                if($dataFactura->tipoPersona ==  2 && isset($comercio->datos->curp) && $comercio->datos->curp != '')
                    $datosEmisorCE['Curp'] = $comercio->datos->curp;

                $nodoEmisorComercioE = $nodoComercioE->addChild("cce$versionComercionXSD:Emisor", true, $datosEmisorCE);

                $emisorCE = $comercio->emisor;
                $domicilioEmisor = array(
                    'Calle' => $emisorCE->calle,
                    'Estado' => $emisorCE->estado,
                    'Pais' => $emisorCE->pais,
                    'CodigoPostal' => $emisorCE->codigoPostal
                );

                if(!empty($emisorCE->nroExterior))
                    $domicilioEmisor['NumeroExterior'] = $emisorCE->nroExterior;

                if(!empty($emisorCE->nroInterior))
                    $domicilioEmisor['NumeroInterior'] = $emisorCE->nroInterior;

                if(!empty($emisorCE->colonia))
                    $domicilioEmisor['Colonia'] = $emisorCE->colonia;

                if(!empty($emisorCE->municipio))
                    $domicilioEmisor['Municipio'] = $emisorCE->municipio;

                $nodoEmisorComercioE->addChild("cce$versionComercionXSD:Domicilio", true, $domicilioEmisor);

                $arrayNumRegIdTrib = array();
                if($comercio->datos->rfcExtranjero != ''){
                    $arrayNumRegIdTrib = array('NumRegIdTrib' =>  $comercio->datos->rfcExtranjero);
                }

                $nodoReceptorComercioE = $nodoComercioE->addChild("cce$versionComercionXSD:Receptor", true, $arrayNumRegIdTrib);

                $receptorCE = $comercio->receptor;
                $domicilioReceptor = array(
                    'Calle' => $receptorCE->calle,
                    'Estado' => $receptorCE->estado,
                    'Pais' => $receptorCE->pais,
                    'CodigoPostal' => $receptorCE->codigoPostal
                );

                if(!empty($receptorCE->nroExterior))
                    $domicilioReceptor['NumeroExterior'] = $receptorCE->nroExterior;

                if(!empty($receptorCE->colonia))
                    $domicilioReceptor['Colonia'] = $receptorCE->colonia;

                if(!empty($receptorCE->nroInterior))
                    $domicilioReceptor['NumeroInterior'] = $receptorCE->nroInterior;

                if(!empty($receptorCE->municipio))
                    $domicilioReceptor['Municipio'] = $receptorCE->municipio;

                $nodoReceptorComercioE->addChild("cce$versionComercionXSD:Domicilio", true, $domicilioReceptor);


                $destinatarioNode = [
                    'NumRegIdTrib' =>  $comercio->datos->rfcExtranjero
                ];

                if(!empty($comercio->datos->nombre))
                    $destinatarioNode['Nombre'] = trim($comercio->datos->nombre);


                $nodoDestinatarioComercioE = $nodoComercioE->addChild("cce$versionComercionXSD:Destinatario", true, ['NumRegIdTrib' =>  $comercio->datos->rfcExtranjero]);
                $nodoDestinatarioComercioE->addChild("cce$versionComercionXSD:Domicilio", true, $domicilioReceptor);


                $nodoMercanciasComercioE = $nodoComercioE->addChild("cce$versionComercionXSD:Mercancias", true);

                if(isset($dataFactura->mercanciasComercio) && count($dataFactura->mercanciasComercio) > 0){
                    foreach ($dataFactura->mercanciasComercio as $mercancia){

                        $conceptoComercio = [
                            'ValorDolares' => Utils::formatNumber ( $mercancia->valorDolares, $dataFactura->precission ),
                            'NoIdentificacion' => $mercancia->noIdentificacion];

                        if(!empty($mercancia->fraccionArancelaria))
                            $conceptoComercio['FraccionArancelaria'] = $mercancia->fraccionArancelaria;

                        if(!empty($mercancia->cantidadAduana))
                            $conceptoComercio['CantidadAduana'] = $mercancia->cantidadAduana;

                        if(!empty($mercancia->unidadAduana))
                            $conceptoComercio['UnidadAduana'] = $mercancia->unidadAduana;

                        if(!empty($mercancia->valorUnitarioAduana))
                            $conceptoComercio['ValorUnitarioAduana'] = Utils::formatNumber($mercancia->valorUnitarioAduana);

                        $nodoMercanciasComercioE->addChild("cce$versionComercionXSD:Mercancia", true, $conceptoComercio);

                    }
                }else{

                    foreach ( $dataFactura->conceptos as $concepto ) {
                        $conceptoComercio = [
                            'CantidadAduana' => $concepto->qtyItem
                        ];
                        if(!empty($concepto->cdgItem))
                            $conceptoComercio['NoIdentificacion'] = $concepto->cdgItem;

                        if(!empty($concepto->fracArancelaria)){

                            $fraccion = FraccionArancelaria::where('codigo','=', $concepto->fracArancelaria)->first();
                            $conceptoComercio['FraccionArancelaria'] = $concepto->fracArancelaria;
                            $conceptoComercio['UnidadAduana'] = $fraccion->unidad;
                        }
                        if($dataFactura->moneda == 'USD'){
                            $conceptoComercio['ValorDolares'] = Utils::formatNumber ( $concepto->prcNetoItem, $dataFactura->precission );
                            $conceptoComercio['ValorUnitarioAduana'] = Utils::formatNumber ( $concepto->montoNetoItem, $dataFactura->precission );
                        }
                        $nodoMercanciasComercioE->addChild("cce$versionComercionXSD:Mercancia", true, $conceptoComercio);
                    }


                }

            }




            if(isset($dataFactura->sello) && $dataFactura->sello != ''){


                $rootNode->setAttribute ( 'Sello', $dataFactura->sello );

                ///Timbre fiscal
                $fechaTimbrado = new \DateTime ( $dataFactura->fechaTimbrado );
                $fechaTimbrado = date_format ( $fechaTimbrado, 'Y-m-d\TH:i:s' );

                if($dataFactura->milisegundos > 0){
                    $fechaTimbrado .= ".".str_pad ( $dataFactura->milisegundos, 3, "0", STR_PAD_LEFT );
                }


                $nodoComplemento->addChild ( 'tfd:TimbreFiscalDigital', '', [
                    'FechaTimbrado' => $fechaTimbrado,
                    "xsi:schemaLocation" => "http://www.sat.gob.mx/TimbreFiscalDigital http://www.sat.gob.mx/sitio_internet/cfd/TimbreFiscalDigital/TimbreFiscalDigitalv11.xsd",
                    'UUID' => $dataFactura->uUID,
                    'NoCertificadoSAT' => $dataFactura->noCertificadoSAT,
                    'SelloCFD' => $dataFactura->selloCFD,
                    'SelloSAT' => $dataFactura->selloSAT,
                    'RfcProvCertif' => $dataFactura->rfcProvCertif,
                    'Version' => '1.1',
                ]);

            }
        }

        if(isset($dataFactura->addenda)){

            $nodoAddenda = $root->addChild ( 'cfdi:Addenda', true);


            if(isset($dataFactura->addenda->inbursa) && $dataFactura->addenda->inbursa != null){

                $inbursa = $dataFactura->addenda->inbursa;
                
                $nodoReferenciaReceptor = $nodoAddenda->addChild('ReferenciaReceptor', true);
                
                $nodoReferenciaReceptor->addChild('Siniestro', true, [
                    'Emisor' => $inbursa->emisor,
                    'Numero' => $inbursa->numero,
                    'Afectado' => $inbursa->afectado
                ]);
                
                $nodoReferenciaReceptor->addChild('Deducible', true, [
                    'Importe' => Utils::formatNumber($inbursa->deducible, 2)
                ]);
                
                $nodoReferenciaReceptor->addChild('Descuento', true, [
                    'Importe' => Utils::formatNumber($inbursa->descuento, 2)
                ]);
                
                $nodoReferenciaReceptor->addChild('TotalManoObra', true, [
                    'Importe' => Utils::formatNumber($inbursa->manaoObra, 2)
                ]);
                
                $nodoReferenciaReceptor->addChild('TotalRefacciones', true, [
                    'Importe' => Utils::formatNumber($inbursa->refacciones, 2)
                ]);
                
                $nodoReferenciaReceptor->addChild('FechaEntregado', true, [
                    'Fecha' => date('d/m/Y', strtotime($inbursa->fechaEntrega))
                ]);

            }else{

                $nodoFondoGarantia = $nodoAddenda->addChild ( 'FondoGarantia:AddendaFondoGarantia', true, [
                    'xmlns:xsi' => 'http://www.w3.org/2001/XMLSchema',
                    "xsi:schemaLocation" => "http://201.148.157.232:81/Content/FondoGarantia http://201.148.157.232:81/Content/FondoGarantia/AddendaFondoGarantia.xsd",
                    'Version' => '1.0'
                ]);

                $nodoFondoGarantia->addChild ( 'FondoGarantia:FondoGarantia', true, [
                    'Importe' => $dataFactura->addenda->Importe,
                    "PeriodoEnMeses" => $dataFactura->addenda->PeriodoEnMeses,
                    'Descripcion' => $dataFactura->addenda->Descripcion
                ]);
            }

        }

        if(!isset($dataFactura->sello)){
            $sello = self::obtenerSello40($xml, $proveedor->keypem);
            $rootNode->setAttribute ( 'Sello', $sello );
        }

        return $xml;
    }


	public static function obtenerSello($xml, $keypemFile){


		// Ruta al archivo XSLT
		$xslFile = "xslt/xslt33/cadenaoriginal_3_3.xslt";

		$xmlTmp = $xml->dom();

		$xsl = new \DOMDocument();
		$xsl->load($xslFile);

		$proc = new XSLTProcessor(new NullCache());
		$proc->importStyleSheet($xsl);
		$cadenaOriginal = trim($proc->transformToXML($xmlTmp));

		file_put_contents("tmp/cadenaoriginal.txt", $cadenaOriginal );
		$pemFileContent = file_get_contents(env('DIR_CERTS')."/$keypemFile");
		$pkeyid = openssl_pkey_get_private($pemFileContent);
		openssl_sign($cadenaOriginal, $keypem, $pkeyid, OPENSSL_ALGO_SHA256);
		$base64 = base64_encode($keypem);
		file_put_contents("tmp/valorDigest.txt", $base64 );
		return $base64;

	}

    public static function obtenerSello40($xml, $keypemFile){


        // Ruta al archivo XSLT
        $xslFile = "xslt/xslt40/cadenaoriginal4.0.xslt";

        $xmlTmp = $xml->dom();

        $xsl = new \DOMDocument();
        $xsl->load($xslFile);

        $proc = new XsltProcessor(new NullCache());
        $proc->importStylesheet($xsl);
        $cadenaOriginal = trim($proc->transformToXML($xmlTmp));

        file_put_contents("tmp/cadenaoriginal.txt", $cadenaOriginal );
        $pemFileContent = file_get_contents(env('DIR_CERTS')."/$keypemFile");
        $pkeyid = openssl_pkey_get_private($pemFileContent);
        openssl_sign($cadenaOriginal, $keypem, $pkeyid, OPENSSL_ALGO_SHA256);
        $base64 = base64_encode($keypem);
        file_put_contents("tmp/valorDigest.txt", $base64 );
        return $base64;

    }

    public static function cast($instance, $className)
    {
        return unserialize(sprintf(
            'O:%d:"%s"%s',
            \strlen($className),
            $className,
            strstr(strstr(serialize($instance), '"'), ':')
        ));
    }

}

?>
